# Generic SysV init.d script
# Copyright (C) 2015 Daniel Boland <d.boland@rocleiden.nl>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

OSNAME=`uname | awk -F_ '{printf $1}'`
RESULT=0
PID=""
STATUS=""
COMMAND="$1"
CMDLIST="help|start|stop|restart|status|graceful|install"

function print_status {
	echo "$0 $COMMAND: $1"
}
function is_running {
	PID=`awk '{print $1; exit}' $PIDFILE 2>/dev/null`
	if ! [ $PID ]; then
		STATUS="$PROGRAM (no pid file) not running"
		return 1
	elif kill -0 $PID 2>/dev/null; then
		STATUS="$PROGRAM (pid $PID) running"
	else
		STATUS="$PROGRAM (pid $PID?) not running"
		return 1
	fi
}
function start_program {
	$PROGFILE $PROGARGS
}
function stop_program {
	kill $PID
}
function restart_program {
	kill -$1 $PID
}
function start_cygwin_service {
	if /usr/bin/cygrunsrv -Q "cyg_$SERVICE" >/dev/null 2>&1; then
		rm -f /var/log/cyg_$SERVICE.log
		/usr/bin/cygrunsrv -S cyg_$SERVICE 2>&1
		result=$?
		cat /var/log/cyg_$SERVICE.log
		return $result
	else
		start_program
	fi
}
function start_service {
	if [ $OSNAME == CYGWIN ]; then
		start_cygwin_service
	else
		start_program
	fi
}
function restart_service {
	if ! is_running; then
		print_status "$PROGRAM not running, trying to start"
		start_service
	else
		restart_program $1
	fi
}
function install_cygwin_service {
	if ! /usr/bin/cygrunsrv -Q "cyg_$SERVICE" 2>/dev/null; then
		STATUS="$SERVICE service installed"
		/usr/bin/cygrunsrv -I "cyg_$SERVICE" -d "CYGWIN $PROGRAM" -t "$CYGTYPE" -o -p "$PROGFILE" -a "$CYGARGS"
	fi
}
function install_service {
	if [ -f /etc/init.d/$SERVICE ]; then
		STATUS="$SERVICE service already installed"
	else
		STATUS="$SERVICE service installed"
		cp $0 /etc/init.d/$SERVICE
	fi
	if [ $OSNAME == CYGWIN ]; then
		install_cygwin_service
	fi
}
function print_syntax {
	echo "Usage: $0 [COMMAND]"
	echo
	echo commands:
	printf -- " start      - start $PROGRAM\n"
	printf -- " stop       - stop $PROGRAM\n"
	printf -- " restart    - restart $PROGRAM by sending a SIGHUP if running or start if \n"
	printf -- "              not running\n"
	printf -- " fullstatus - dump a full status screen; requires lynx and mod_status enabled\n"
	printf -- " status     - dump a short status screen; requires lynx and mod_status enabled\n"
	printf -- " graceful   - do a graceful restart by sending a SIGUSR1 or start if not running\n"
	printf -- " help       - this screen\n"
}
function do_help {
	print_syntax
}
function do_start {
	if is_running; then
		print_status "$STATUS"
	elif start_service; then
		print_status "$PROGRAM started"
	else
		RESULT=$?
		print_status "$PROGRAM could not be started ($RESULT)"
	fi
}
function do_stop {
	if ! is_running; then
		print_status "$STATUS"
	elif stop_program; then
		print_status "$PROGRAM stopped"
	else
		RESULT=$?
		print_status "$PROGRAM could not be stopped ($RESULT)"
	fi
}
function do_restart {
	if restart_service HUP; then
		print_status "$PROGRAM restarted"
	else
		RESULT=$?
		print_status "$PROGRAM could not be restarted ($RESULT)"
	fi
}
function do_status {
	is_running || RESULT=1
	print_status "$STATUS"
}
function do_graceful {
	if restart_service USR1; then
		print_status "$PROGRAM restarted"
	else
		RESULT=$?
		print_status "$PROGRAM could not be restarted gracefully ($RESULT)"
	fi
}
function do_install {
	if install_service; then
		print_status "$STATUS"
	else
		RESULT=$?
		print_status "$SERVICE service could not be installed ($RESULT)"
	fi
}
function do_command {
	if ! [ "$COMMAND" ]; then
		echo "Usage: $0 {$CMDLIST}"
	elif do_$COMMAND 2>/dev/null; then
		exit $RESULT
	else
		print_status "Command not implemented"
		echo "Usage: $0 {$CMDLIST}"
		exit 1
	fi
}
