/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Timpctl    timpctl         Time percentiles
      Hourpctl   hourpctl        Hourly percentiles
      Daypctl    daypctl         Daily percentiles
      Monpctl    monpctl         Monthly percentiles
      Yearpctl   yearpctl        Yearly percentiles
*/

#include <cdi.h>

#include "util_date.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "param_conversion.h"
#include "percentiles_hist.h"
#include "percentiles.h"
#include "datetime.h"

static void
timpctl(int operatorID)
{
  TimeStat timestat_date = TimeStat::MEAN;
  char indate1[DATE_LEN + 1], indate2[DATE_LEN + 1];
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  operatorInputArg("percentile number");
  const auto pn = parameter2double(cdoOperatorArgv(0));
  percentile_check_number(pn);

  const int cmplen = DATE_LEN - cdoOperatorF2(operatorID);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);
  const auto streamID3 = cdoOpenRead(2);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = cdoStreamInqVlist(streamID3);
  const auto vlistID4 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);
  vlistCompare(vlistID1, vlistID3, CMP_ALL);

  if (cdoOperatorF2(operatorID) == 16) vlistDefNtsteps(vlistID4, 1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = vlistInqTaxis(vlistID3);
  /* TODO - check that time axes 2 and 3 are equal */

  const auto taxisID4 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID4);
  vlistDefTaxis(vlistID4, taxisID4);

  const auto streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  FieldVector2D vars1;
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);
  HistogramSet hset(nvars);

  for (varID = 0; varID < nvars; varID++)
    {
      auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
      auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      hset.createVarLevels(varID, nlevels, gridsize);
    }

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      nrecs = cdoStreamInqTimestep(streamID2, otsID);
      if (nrecs != cdoStreamInqTimestep(streamID3, otsID))
        cdoAbort("Number of records at time step %d of %s and %s differ!", otsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      const auto vdate2 = taxisInqVdate(taxisID2);
      const auto vtime2 = taxisInqVtime(taxisID2);
      const auto vdate3 = taxisInqVdate(taxisID3);
      const auto vtime3 = taxisInqVtime(taxisID3);
      if (vdate2 != vdate3 || vtime2 != vtime3)
        cdoAbort("Verification dates at time step %d of %s and %s differ!", otsID + 1, cdoGetStreamName(1), cdoGetStreamName(2));

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars1[varID][levelID].vec.data(), &nmiss);
          vars1[varID][levelID].nmiss = nmiss;
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, field.vec.data(), &field.nmiss);
          field.grid = vars1[varID][levelID].grid;
          field.missval = vars1[varID][levelID].missval;

          hset.defVarLevelBounds(varID, levelID, vars1[varID][levelID], field);
        }

      int nsets = 0;
      while (nrecs && (nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          dtlist.taxisInqTimestep(taxisID1, nsets);
          const auto vdate1 = dtlist.getVdate(nsets);
          const auto vtime1 = dtlist.getVtime(nsets);

          if (nsets == 0) SET_DATE(indate2, vdate1, vtime1);
          SET_DATE(indate1, vdate1, vtime1);

          if (DATE_IS_NEQ(indate1, indate2, cmplen)) break;

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);
              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
                }
              cdoReadRecord(streamID1, vars1[varID][levelID].vec.data(), &nmiss);
              vars1[varID][levelID].nmiss = nmiss;

              hset.addSubVarLevelValues(varID, levelID, vars1[varID][levelID], 1, FIELD_VEC);
            }

          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

          const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
          for (levelID = 0; levelID < nlevels; levelID++) hset.getVarLevelPercentiles(vars1[varID][levelID], varID, levelID, pn, FIELD_VEC);
        }

      dtlist.statTaxisDefTimestep(taxisID4, nsets);
      cdoDefTimestep(streamID4, otsID);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          cdoDefRecord(streamID4, varID, levelID);
          cdoWriteRecord(streamID4, vars1[varID][levelID].vec.data(), vars1[varID][levelID].nmiss);
        }

      if (nrecs == 0) break;
      otsID++;
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);
}

void *
Timpctl(void *process)
{
  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("timpctl",  func_pctl, 31, nullptr);
  cdoOperatorAdd("yearpctl", func_pctl, 10, nullptr);
  cdoOperatorAdd("monpctl",  func_pctl,  8, nullptr);
  cdoOperatorAdd("daypctl",  func_pctl,  6, nullptr);
  cdoOperatorAdd("hourpctl", func_pctl,  4, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();

  timpctl(operatorID);

  cdoFinish();

  return nullptr;
}
