// 2007-02-04  Edward Smith-Rowland <3dw4rd@verizon.net>
//
// Copyright (C) 2007-2016 Free Software Foundation, Inc.
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3, or (at your option)
// any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING3.  If not see
// <http://www.gnu.org/licenses/>.

//  sph_legendre


//  Compare against values generated by the GNU Scientific Library.
//  The GSL can be found on the web: http://www.gnu.org/software/gsl/

#include <tr1/cmath>
#if defined(__TEST_DEBUG)
#include <iostream>
#define VERIFY(A) \
if (!(A)) \
  { \
    std::cout << "line " << __LINE__ \
      << "  max_abs_frac = " << max_abs_frac \
      << std::endl; \
  }
#else
#include <testsuite_hooks.h>
#endif
#include "../testcase.h"


// Test data for l=0, m=0.
testcase_sph_legendre<double> data001[] = {
  { 0.28209479177387814, 0, 0, 
          0.0000000000000000 },
  { 0.28209479177387814, 0, 0, 
          0.15707963267948966 },
  { 0.28209479177387814, 0, 0, 
          0.31415926535897931 },
  { 0.28209479177387814, 0, 0, 
          0.47123889803846897 },
  { 0.28209479177387814, 0, 0, 
          0.62831853071795862 },
  { 0.28209479177387814, 0, 0, 
          0.78539816339744828 },
  { 0.28209479177387814, 0, 0, 
          0.94247779607693793 },
  { 0.28209479177387814, 0, 0, 
          1.0995574287564276 },
  { 0.28209479177387814, 0, 0, 
          1.2566370614359172 },
  { 0.28209479177387814, 0, 0, 
          1.4137166941154069 },
  { 0.28209479177387814, 0, 0, 
          1.5707963267948966 },
  { 0.28209479177387814, 0, 0, 
          1.7278759594743860 },
  { 0.28209479177387814, 0, 0, 
          1.8849555921538759 },
  { 0.28209479177387814, 0, 0, 
          2.0420352248333655 },
  { 0.28209479177387814, 0, 0, 
          2.1991148575128552 },
  { 0.28209479177387814, 0, 0, 
          2.3561944901923448 },
  { 0.28209479177387814, 0, 0, 
          2.5132741228718345 },
  { 0.28209479177387814, 0, 0, 
          2.6703537555513241 },
  { 0.28209479177387814, 0, 0, 
          2.8274333882308138 },
  { 0.28209479177387814, 0, 0, 
          2.9845130209103035 },
  { 0.28209479177387814, 0, 0, 
          3.1415926535897931 },
};

// Test function for l=0, m=0.
template <typename Tp>
void test001()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data001)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data001[i].l), Tp(data001[i].m),
                   Tp(data001[i].theta));
      const Tp f0 = data001[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=1, m=0.
testcase_sph_legendre<double> data002[] = {
  { 0.48860251190291992, 1, 0, 
          0.0000000000000000 },
  { 0.48258700419201100, 1, 0, 
          0.15707963267948966 },
  { 0.46468860282345231, 1, 0, 
          0.31415926535897931 },
  { 0.43534802584032634, 1, 0, 
          0.47123889803846897 },
  { 0.39528773562374975, 1, 0, 
          0.62831853071795862 },
  { 0.34549414947133544, 1, 0, 
          0.78539816339744828 },
  { 0.28719335072959390, 1, 0, 
          0.94247779607693793 },
  { 0.22182089855280451, 1, 0, 
          1.0995574287564276 },
  { 0.15098647967228984, 1, 0, 
          1.2566370614359172 },
  { 0.076434272566846345, 1, 0, 
          1.4137166941154069 },
  { 2.9918275112863369e-17, 1, 0, 
          1.5707963267948966 },
  { -0.076434272566846179, 1, 0, 
          1.7278759594743860 },
  { -0.15098647967228976, 1, 0, 
          1.8849555921538759 },
  { -0.22182089855280446, 1, 0, 
          2.0420352248333655 },
  { -0.28719335072959390, 1, 0, 
          2.1991148575128552 },
  { -0.34549414947133544, 1, 0, 
          2.3561944901923448 },
  { -0.39528773562374969, 1, 0, 
          2.5132741228718345 },
  { -0.43534802584032628, 1, 0, 
          2.6703537555513241 },
  { -0.46468860282345231, 1, 0, 
          2.8274333882308138 },
  { -0.48258700419201095, 1, 0, 
          2.9845130209103035 },
  { -0.48860251190291992, 1, 0, 
          3.1415926535897931 },
};

// Test function for l=1, m=0.
template <typename Tp>
void test002()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data002)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data002[i].l), Tp(data002[i].m),
                   Tp(data002[i].theta));
      const Tp f0 = data002[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=1, m=1.
testcase_sph_legendre<double> data003[] = {
  { 0.0000000000000000, 1, 1, 
          0.0000000000000000 },
  { -0.054047192447077917, 1, 1, 
          0.15707963267948966 },
  { -0.10676356364376104, 1, 1, 
          0.31415926535897931 },
  { -0.15685106157558129, 1, 1, 
          0.47123889803846897 },
  { -0.20307636581258243, 1, 1, 
          0.62831853071795862 },
  { -0.24430125595146013, 1, 1, 
          0.78539816339744828 },
  { -0.27951063837942880, 1, 1, 
          0.94247779607693793 },
  { -0.30783754124787122, 1, 1, 
          1.0995574287564276 },
  { -0.32858446219656551, 1, 1, 
          1.2566370614359172 },
  { -0.34124054317667202, 1, 1, 
          1.4137166941154069 },
  { -0.34549414947133567, 1, 1, 
          1.5707963267948966 },
  { -0.34124054317667202, 1, 1, 
          1.7278759594743860 },
  { -0.32858446219656556, 1, 1, 
          1.8849555921538759 },
  { -0.30783754124787127, 1, 1, 
          2.0420352248333655 },
  { -0.27951063837942880, 1, 1, 
          2.1991148575128552 },
  { -0.24430125595146013, 1, 1, 
          2.3561944901923448 },
  { -0.20307636581258248, 1, 1, 
          2.5132741228718345 },
  { -0.15685106157558140, 1, 1, 
          2.6703537555513241 },
  { -0.10676356364376104, 1, 1, 
          2.8274333882308138 },
  { -0.054047192447078167, 1, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 1, 1, 
          3.1415926535897931 },
};

// Test function for l=1, m=1.
template <typename Tp>
void test003()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data003)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data003[i].l), Tp(data003[i].m),
                   Tp(data003[i].theta));
      const Tp f0 = data003[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=2, m=0.
testcase_sph_legendre<double> data004[] = {
  { 0.63078313050504009, 2, 0, 
          0.0000000000000000 },
  { 0.60762858760316607, 2, 0, 
          0.15707963267948966 },
  { 0.54043148688396569, 2, 0, 
          0.31415926535897931 },
  { 0.43576954875556589, 2, 0, 
          0.47123889803846897 },
  { 0.30388781294457579, 2, 0, 
          0.62831853071795862 },
  { 0.15769578262625994, 2, 0, 
          0.78539816339744828 },
  { 0.011503752307944235, 2, 0, 
          0.94247779607693793 },
  { -0.12037798350304565, 2, 0, 
          1.0995574287564276 },
  { -0.22503992163144573, 2, 0, 
          1.2566370614359172 },
  { -0.29223702235064597, 2, 0, 
          1.4137166941154069 },
  { -0.31539156525252005, 2, 0, 
          1.5707963267948966 },
  { -0.29223702235064608, 2, 0, 
          1.7278759594743860 },
  { -0.22503992163144584, 2, 0, 
          1.8849555921538759 },
  { -0.12037798350304577, 2, 0, 
          2.0420352248333655 },
  { 0.011503752307944235, 2, 0, 
          2.1991148575128552 },
  { 0.15769578262625994, 2, 0, 
          2.3561944901923448 },
  { 0.30388781294457567, 2, 0, 
          2.5132741228718345 },
  { 0.43576954875556562, 2, 0, 
          2.6703537555513241 },
  { 0.54043148688396569, 2, 0, 
          2.8274333882308138 },
  { 0.60762858760316585, 2, 0, 
          2.9845130209103035 },
  { 0.63078313050504009, 2, 0, 
          3.1415926535897931 },
};

// Test function for l=2, m=0.
template <typename Tp>
void test004()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data004)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data004[i].l), Tp(data004[i].m),
                   Tp(data004[i].theta));
      const Tp f0 = data004[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=2, m=1.
testcase_sph_legendre<double> data005[] = {
  { 0.0000000000000000, 2, 1, 
          0.0000000000000000 },
  { -0.11936529291378727, 2, 1, 
          0.15707963267948966 },
  { -0.22704627929027449, 2, 1, 
          0.31415926535897931 },
  { -0.31250239392538215, 2, 1, 
          0.47123889803846897 },
  { -0.36736859691086526, 2, 1, 
          0.62831853071795862 },
  { -0.38627420202318979, 2, 1, 
          0.78539816339744828 },
  { -0.36736859691086526, 2, 1, 
          0.94247779607693793 },
  { -0.31250239392538226, 2, 1, 
          1.0995574287564276 },
  { -0.22704627929027435, 2, 1, 
          1.2566370614359172 },
  { -0.11936529291378740, 2, 1, 
          1.4137166941154069 },
  { -4.7304946510089748e-17, 2, 1, 
          1.5707963267948966 },
  { 0.11936529291378714, 2, 1, 
          1.7278759594743860 },
  { 0.22704627929027429, 2, 1, 
          1.8849555921538759 },
  { 0.31250239392538226, 2, 1, 
          2.0420352248333655 },
  { 0.36736859691086526, 2, 1, 
          2.1991148575128552 },
  { 0.38627420202318979, 2, 1, 
          2.3561944901923448 },
  { 0.36736859691086526, 2, 1, 
          2.5132741228718345 },
  { 0.31250239392538232, 2, 1, 
          2.6703537555513241 },
  { 0.22704627929027449, 2, 1, 
          2.8274333882308138 },
  { 0.11936529291378781, 2, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 2, 1, 
          3.1415926535897931 },
};

// Test function for l=2, m=1.
template <typename Tp>
void test005()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data005)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data005[i].l), Tp(data005[i].m),
                   Tp(data005[i].theta));
      const Tp f0 = data005[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=2, m=2.
testcase_sph_legendre<double> data006[] = {
  { 0.0000000000000000, 2, 2, 
          0.0000000000000000 },
  { 0.0094528025561622549, 2, 2, 
          0.15707963267948966 },
  { 0.036885904048903795, 2, 2, 
          0.31415926535897931 },
  { 0.079613961366457681, 2, 2, 
          0.47123889803846897 },
  { 0.13345445455470123, 2, 2, 
          0.62831853071795862 },
  { 0.19313710101159492, 2, 2, 
          0.78539816339744828 },
  { 0.25281974746848851, 2, 2, 
          0.94247779607693793 },
  { 0.30666024065673203, 2, 2, 
          1.0995574287564276 },
  { 0.34938829797428600, 2, 2, 
          1.2566370614359172 },
  { 0.37682139946702747, 2, 2, 
          1.4137166941154069 },
  { 0.38627420202318979, 2, 2, 
          1.5707963267948966 },
  { 0.37682139946702753, 2, 2, 
          1.7278759594743860 },
  { 0.34938829797428606, 2, 2, 
          1.8849555921538759 },
  { 0.30666024065673209, 2, 2, 
          2.0420352248333655 },
  { 0.25281974746848851, 2, 2, 
          2.1991148575128552 },
  { 0.19313710101159492, 2, 2, 
          2.3561944901923448 },
  { 0.13345445455470126, 2, 2, 
          2.5132741228718345 },
  { 0.079613961366457764, 2, 2, 
          2.6703537555513241 },
  { 0.036885904048903795, 2, 2, 
          2.8274333882308138 },
  { 0.0094528025561623433, 2, 2, 
          2.9845130209103035 },
  { 0.0000000000000000, 2, 2, 
          3.1415926535897931 },
};

// Test function for l=2, m=2.
template <typename Tp>
void test006()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data006)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data006[i].l), Tp(data006[i].m),
                   Tp(data006[i].theta));
      const Tp f0 = data006[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=0.
testcase_sph_legendre<double> data007[] = {
  { 0.93560257962738880, 5, 0, 
          0.0000000000000000 },
  { 0.77014422942080019, 5, 0, 
          0.15707963267948966 },
  { 0.35892185032365215, 5, 0, 
          0.31415926535897931 },
  { -0.090214932090594183, 5, 0, 
          0.47123889803846897 },
  { -0.36214460396518888, 5, 0, 
          0.62831853071795862 },
  { -0.35145955579226890, 5, 0, 
          0.78539816339744828 },
  { -0.11441703594725168, 5, 0, 
          0.94247779607693793 },
  { 0.17248966720808098, 5, 0, 
          1.0995574287564276 },
  { 0.32128384287200523, 5, 0, 
          1.2566370614359172 },
  { 0.24377632246714948, 5, 0, 
          1.4137166941154069 },
  { 1.0741712853887702e-16, 5, 0, 
          1.5707963267948966 },
  { -0.24377632246714906, 5, 0, 
          1.7278759594743860 },
  { -0.32128384287200534, 5, 0, 
          1.8849555921538759 },
  { -0.17248966720808118, 5, 0, 
          2.0420352248333655 },
  { 0.11441703594725168, 5, 0, 
          2.1991148575128552 },
  { 0.35145955579226890, 5, 0, 
          2.3561944901923448 },
  { 0.36214460396518910, 5, 0, 
          2.5132741228718345 },
  { 0.090214932090594724, 5, 0, 
          2.6703537555513241 },
  { -0.35892185032365215, 5, 0, 
          2.8274333882308138 },
  { -0.77014422942079797, 5, 0, 
          2.9845130209103035 },
  { -0.93560257962738880, 5, 0, 
          3.1415926535897931 },
};

// Test function for l=5, m=0.
template <typename Tp>
void test007()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data007)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data007[i].l), Tp(data007[i].m),
                   Tp(data007[i].theta));
      const Tp f0 = data007[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=1.
testcase_sph_legendre<double> data008[] = {
  { 0.0000000000000000, 5, 1, 
          0.0000000000000000 },
  { -0.36712373713318258, 5, 1, 
          0.15707963267948966 },
  { -0.54610329010534753, 5, 1, 
          0.31415926535897931 },
  { -0.45381991493631785, 5, 1, 
          0.47123889803846897 },
  { -0.15679720635769953, 5, 1, 
          0.62831853071795862 },
  { 0.16985499419838637, 5, 1, 
          0.78539816339744828 },
  { 0.34468004499725180, 5, 1, 
          0.94247779607693793 },
  { 0.28349471119605985, 5, 1, 
          1.0995574287564276 },
  { 0.044286619339675856, 5, 1, 
          1.2566370614359172 },
  { -0.21193784177193470, 5, 1, 
          1.4137166941154069 },
  { -0.32028164857621527, 5, 1, 
          1.5707963267948966 },
  { -0.21193784177193514, 5, 1, 
          1.7278759594743860 },
  { 0.044286619339675606, 5, 1, 
          1.8849555921538759 },
  { 0.28349471119605979, 5, 1, 
          2.0420352248333655 },
  { 0.34468004499725180, 5, 1, 
          2.1991148575128552 },
  { 0.16985499419838637, 5, 1, 
          2.3561944901923448 },
  { -0.15679720635769920, 5, 1, 
          2.5132741228718345 },
  { -0.45381991493631757, 5, 1, 
          2.6703537555513241 },
  { -0.54610329010534753, 5, 1, 
          2.8274333882308138 },
  { -0.36712373713318397, 5, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 5, 1, 
          3.1415926535897931 },
};

// Test function for l=5, m=1.
template <typename Tp>
void test008()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data008)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data008[i].l), Tp(data008[i].m),
                   Tp(data008[i].theta));
      const Tp f0 = data008[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=2.
testcase_sph_legendre<double> data009[] = {
  { 0.0000000000000000, 5, 2, 
          0.0000000000000000 },
  { 0.078919441745546146, 5, 2, 
          0.15707963267948966 },
  { 0.26373799140437981, 5, 2, 
          0.31415926535897931 },
  { 0.43002359842080096, 5, 2, 
          0.47123889803846897 },
  { 0.45642486439050994, 5, 2, 
          0.62831853071795862 },
  { 0.29959604906083276, 5, 2, 
          0.78539816339744828 },
  { 0.023781239849532242, 5, 2, 
          0.94247779607693793 },
  { -0.23313989334673815, 5, 2, 
          1.0995574287564276 },
  { -0.33799912776303714, 5, 2, 
          1.2566370614359172 },
  { -0.23964508489529743, 5, 2, 
          1.4137166941154069 },
  { -1.0377480524338170e-16, 5, 2, 
          1.5707963267948966 },
  { 0.23964508489529704, 5, 2, 
          1.7278759594743860 },
  { 0.33799912776303714, 5, 2, 
          1.8849555921538759 },
  { 0.23313989334673837, 5, 2, 
          2.0420352248333655 },
  { -0.023781239849532242, 5, 2, 
          2.1991148575128552 },
  { -0.29959604906083276, 5, 2, 
          2.3561944901923448 },
  { -0.45642486439050978, 5, 2, 
          2.5132741228718345 },
  { -0.43002359842080118, 5, 2, 
          2.6703537555513241 },
  { -0.26373799140437981, 5, 2, 
          2.8274333882308138 },
  { -0.078919441745546867, 5, 2, 
          2.9845130209103035 },
  { 0.0000000000000000, 5, 2, 
          3.1415926535897931 },
};

// Test function for l=5, m=2.
template <typename Tp>
void test009()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data009)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data009[i].l), Tp(data009[i].m),
                   Tp(data009[i].theta));
      const Tp f0 = data009[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=5, m=5.
testcase_sph_legendre<double> data010[] = {
  { 0.0000000000000000, 5, 5, 
          0.0000000000000000 },
  { -4.3481439097909148e-05, 5, 5, 
          0.15707963267948966 },
  { -0.0013078367086431812, 5, 5, 
          0.31415926535897931 },
  { -0.0089510818191922761, 5, 5, 
          0.47123889803846897 },
  { -0.032563803777573896, 5, 5, 
          0.62831853071795862 },
  { -0.082047757105021310, 5, 5, 
          0.78539816339744828 },
  { -0.16085328164143819, 5, 5, 
          0.94247779607693793 },
  { -0.26064303436645375, 5, 5, 
          1.0995574287564276 },
  { -0.36113811790820566, 5, 5, 
          1.2566370614359172 },
  { -0.43625592459446139, 5, 5, 
          1.4137166941154069 },
  { -0.46413220344085809, 5, 5, 
          1.5707963267948966 },
  { -0.43625592459446155, 5, 5, 
          1.7278759594743860 },
  { -0.36113811790820577, 5, 5, 
          1.8849555921538759 },
  { -0.26064303436645381, 5, 5, 
          2.0420352248333655 },
  { -0.16085328164143819, 5, 5, 
          2.1991148575128552 },
  { -0.082047757105021310, 5, 5, 
          2.3561944901923448 },
  { -0.032563803777573924, 5, 5, 
          2.5132741228718345 },
  { -0.0089510818191923004, 5, 5, 
          2.6703537555513241 },
  { -0.0013078367086431812, 5, 5, 
          2.8274333882308138 },
  { -4.3481439097910151e-05, 5, 5, 
          2.9845130209103035 },
  { 0.0000000000000000, 5, 5, 
          3.1415926535897931 },
};

// Test function for l=5, m=5.
template <typename Tp>
void test010()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data010)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data010[i].l), Tp(data010[i].m),
                   Tp(data010[i].theta));
      const Tp f0 = data010[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=0.
testcase_sph_legendre<double> data011[] = {
  { 1.2927207364566027, 10, 0, 
          0.0000000000000000 },
  { 0.55288895150522632, 10, 0, 
          0.15707963267948966 },
  { -0.44874428379711506, 10, 0, 
          0.31415926535897931 },
  { -0.25532095827149692, 10, 0, 
          0.47123889803846897 },
  { 0.36625249688013967, 10, 0, 
          0.62831853071795862 },
  { 0.14880806329084145, 10, 0, 
          0.78539816339744828 },
  { -0.33533356797848746, 10, 0, 
          0.94247779607693793 },
  { -0.080639967662335818, 10, 0, 
          1.0995574287564276 },
  { 0.32197986450174521, 10, 0, 
          1.2566370614359172 },
  { 0.025713542103667848, 10, 0, 
          1.4137166941154069 },
  { -0.31813049373736707, 10, 0, 
          1.5707963267948966 },
  { 0.025713542103666668, 10, 0, 
          1.7278759594743860 },
  { 0.32197986450174532, 10, 0, 
          1.8849555921538759 },
  { -0.080639967662335416, 10, 0, 
          2.0420352248333655 },
  { -0.33533356797848746, 10, 0, 
          2.1991148575128552 },
  { 0.14880806329084145, 10, 0, 
          2.3561944901923448 },
  { 0.36625249688013994, 10, 0, 
          2.5132741228718345 },
  { -0.25532095827149576, 10, 0, 
          2.6703537555513241 },
  { -0.44874428379711506, 10, 0, 
          2.8274333882308138 },
  { 0.55288895150521977, 10, 0, 
          2.9845130209103035 },
  { 1.2927207364566027, 10, 0, 
          3.1415926535897931 },
};

// Test function for l=10, m=0.
template <typename Tp>
void test011()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data011)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data011[i].l), Tp(data011[i].m),
                   Tp(data011[i].theta));
      const Tp f0 = data011[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=1.
testcase_sph_legendre<double> data012[] = {
  { 0.0000000000000000, 10, 1, 
          0.0000000000000000 },
  { -0.74373723919063905, 10, 1, 
          0.15707963267948966 },
  { -0.29035110456209551, 10, 1, 
          0.31415926535897931 },
  { 0.42219282075271497, 10, 1, 
          0.47123889803846897 },
  { 0.17109256898931269, 10, 1, 
          0.62831853071795862 },
  { -0.35583574648544292, 10, 1, 
          0.78539816339744828 },
  { -0.10089212303543979, 10, 1, 
          0.94247779607693793 },
  { 0.32997652649321085, 10, 1, 
          1.0995574287564276 },
  { 0.047416376890033113, 10, 1, 
          1.2566370614359172 },
  { -0.31999356750295654, 10, 1, 
          1.4137166941154069 },
  { -2.0430664782290766e-16, 10, 1, 
          1.5707963267948966 },
  { 0.31999356750295660, 10, 1, 
          1.7278759594743860 },
  { -0.047416376890032523, 10, 1, 
          1.8849555921538759 },
  { -0.32997652649321102, 10, 1, 
          2.0420352248333655 },
  { 0.10089212303543979, 10, 1, 
          2.1991148575128552 },
  { 0.35583574648544292, 10, 1, 
          2.3561944901923448 },
  { -0.17109256898931186, 10, 1, 
          2.5132741228718345 },
  { -0.42219282075271553, 10, 1, 
          2.6703537555513241 },
  { 0.29035110456209551, 10, 1, 
          2.8274333882308138 },
  { 0.74373723919064028, 10, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 10, 1, 
          3.1415926535897931 },
};

// Test function for l=10, m=1.
template <typename Tp>
void test012()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data012)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data012[i].l), Tp(data012[i].m),
                   Tp(data012[i].theta));
      const Tp f0 = data012[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=2.
testcase_sph_legendre<double> data013[] = {
  { 0.0000000000000000, 10, 2, 
          0.0000000000000000 },
  { 0.34571695599980284, 10, 2, 
          0.15707963267948966 },
  { 0.62485535978198059, 10, 2, 
          0.31415926535897931 },
  { 0.098210039644716363, 10, 2, 
          0.47123889803846897 },
  { -0.41494799233049656, 10, 2, 
          0.62831853071795862 },
  { -0.081698973831472149, 10, 2, 
          0.78539816339744828 },
  { 0.35253132222271272, 10, 2, 
          0.94247779607693793 },
  { 0.049026298555981063, 10, 2, 
          1.0995574287564276 },
  { -0.32791246874130797, 10, 2, 
          1.2566370614359172 },
  { -0.016196782433946910, 10, 2, 
          1.4137166941154069 },
  { 0.32106263400438328, 10, 2, 
          1.5707963267948966 },
  { -0.016196782433945765, 10, 2, 
          1.7278759594743860 },
  { -0.32791246874130797, 10, 2, 
          1.8849555921538759 },
  { 0.049026298555980702, 10, 2, 
          2.0420352248333655 },
  { 0.35253132222271272, 10, 2, 
          2.1991148575128552 },
  { -0.081698973831472149, 10, 2, 
          2.3561944901923448 },
  { -0.41494799233049667, 10, 2, 
          2.5132741228718345 },
  { 0.098210039644714753, 10, 2, 
          2.6703537555513241 },
  { 0.62485535978198059, 10, 2, 
          2.8274333882308138 },
  { 0.34571695599980551, 10, 2, 
          2.9845130209103035 },
  { 0.0000000000000000, 10, 2, 
          3.1415926535897931 },
};

// Test function for l=10, m=2.
template <typename Tp>
void test013()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data013)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data013[i].l), Tp(data013[i].m),
                   Tp(data013[i].theta));
      const Tp f0 = data013[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=10, m=5.
testcase_sph_legendre<double> data014[] = {
  { 0.0000000000000000, 10, 5, 
          0.0000000000000000 },
  { -0.0030300124052750187, 10, 5, 
          0.15707963267948966 },
  { -0.070348585248056830, 10, 5, 
          0.31415926535897931 },
  { -0.30055029290703639, 10, 5, 
          0.47123889803846897 },
  { -0.49987818144009166, 10, 5, 
          0.62831853071795862 },
  { -0.28108771757150086, 10, 5, 
          0.78539816339744828 },
  { 0.22068081187249308, 10, 5, 
          0.94247779607693793 },
  { 0.33689502212592115, 10, 5, 
          1.0995574287564276 },
  { -0.086095515520763985, 10, 5, 
          1.2566370614359172 },
  { -0.33935827318511558, 10, 5, 
          1.4137166941154069 },
  { -1.9213014340664578e-16, 10, 5, 
          1.5707963267948966 },
  { 0.33935827318511552, 10, 5, 
          1.7278759594743860 },
  { 0.086095515520764526, 10, 5, 
          1.8849555921538759 },
  { -0.33689502212592098, 10, 5, 
          2.0420352248333655 },
  { -0.22068081187249308, 10, 5, 
          2.1991148575128552 },
  { 0.28108771757150086, 10, 5, 
          2.3561944901923448 },
  { 0.49987818144009155, 10, 5, 
          2.5132741228718345 },
  { 0.30055029290703678, 10, 5, 
          2.6703537555513241 },
  { 0.070348585248056830, 10, 5, 
          2.8274333882308138 },
  { 0.0030300124052750855, 10, 5, 
          2.9845130209103035 },
  { 0.0000000000000000, 10, 5, 
          3.1415926535897931 },
};

// Test function for l=10, m=5.
template <typename Tp>
void test014()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data014)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data014[i].l), Tp(data014[i].m),
                   Tp(data014[i].theta));
      const Tp f0 = data014[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=10, m=10.
testcase_sph_legendre<double> data015[] = {
  { 0.0000000000000000, 10, 10, 
          0.0000000000000000 },
  { 4.7624282733343473e-09, 10, 10, 
          0.15707963267948966 },
  { 4.3085156534549772e-06, 10, 10, 
          0.31415926535897931 },
  { 0.00020182347649472368, 10, 10, 
          0.47123889803846897 },
  { 0.0026711045506511684, 10, 10, 
          0.62831853071795862 },
  { 0.016957196623256943, 10, 10, 
          0.78539816339744828 },
  { 0.065174916004990341, 10, 10, 
          0.94247779607693793 },
  { 0.17112476903017843, 10, 10, 
          1.0995574287564276 },
  { 0.32852414199733548, 10, 10, 
          1.2566370614359172 },
  { 0.47940582314838287, 10, 10, 
          1.4137166941154069 },
  { 0.54263029194422152, 10, 10, 
          1.5707963267948966 },
  { 0.47940582314838309, 10, 10, 
          1.7278759594743860 },
  { 0.32852414199733571, 10, 10, 
          1.8849555921538759 },
  { 0.17112476903017854, 10, 10, 
          2.0420352248333655 },
  { 0.065174916004990341, 10, 10, 
          2.1991148575128552 },
  { 0.016957196623256943, 10, 10, 
          2.3561944901923448 },
  { 0.0026711045506511706, 10, 10, 
          2.5132741228718345 },
  { 0.00020182347649472493, 10, 10, 
          2.6703537555513241 },
  { 4.3085156534549772e-06, 10, 10, 
          2.8274333882308138 },
  { 4.7624282733345673e-09, 10, 10, 
          2.9845130209103035 },
  { 0.0000000000000000, 10, 10, 
          3.1415926535897931 },
};

// Test function for l=10, m=10.
template <typename Tp>
void test015()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data015)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data015[i].l), Tp(data015[i].m),
                   Tp(data015[i].theta));
      const Tp f0 = data015[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=0.
testcase_sph_legendre<double> data016[] = {
  { 1.8062879984608917, 20, 0, 
          0.0000000000000000 },
  { -0.58906549291415966, 20, 0, 
          0.15707963267948966 },
  { 0.45624611402342408, 20, 0, 
          0.31415926535897931 },
  { -0.39955402700466724, 20, 0, 
          0.47123889803846897 },
  { 0.36818552901640750, 20, 0, 
          0.62831853071795862 },
  { -0.34873131330857787, 20, 0, 
          0.78539816339744828 },
  { 0.33600882829186501, 20, 0, 
          0.94247779607693793 },
  { -0.32759286308122904, 20, 0, 
          1.0995574287564276 },
  { 0.32222458068091320, 20, 0, 
          1.2566370614359172 },
  { -0.31922731037135960, 20, 0, 
          1.4137166941154069 },
  { 0.31826262039531755, 20, 0, 
          1.5707963267948966 },
  { -0.31922731037135965, 20, 0, 
          1.7278759594743860 },
  { 0.32222458068091336, 20, 0, 
          1.8849555921538759 },
  { -0.32759286308122937, 20, 0, 
          2.0420352248333655 },
  { 0.33600882829186501, 20, 0, 
          2.1991148575128552 },
  { -0.34873131330857787, 20, 0, 
          2.3561944901923448 },
  { 0.36818552901640839, 20, 0, 
          2.5132741228718345 },
  { -0.39955402700466852, 20, 0, 
          2.6703537555513241 },
  { 0.45624611402342408, 20, 0, 
          2.8274333882308138 },
  { -0.58906549291416699, 20, 0, 
          2.9845130209103035 },
  { 1.8062879984608917, 20, 0, 
          3.1415926535897931 },
};

// Test function for l=20, m=0.
template <typename Tp>
void test016()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data016)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data016[i].l), Tp(data016[i].m),
                   Tp(data016[i].theta));
      const Tp f0 = data016[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=1.
testcase_sph_legendre<double> data017[] = {
  { 0.0000000000000000, 20, 1, 
          0.0000000000000000 },
  { -0.45905213045060206, 20, 1, 
          0.15707963267948966 },
  { 0.31166370423309170, 20, 1, 
          0.31415926535897931 },
  { -0.23278757741246814, 20, 1, 
          0.47123889803846897 },
  { 0.17937240823504183, 20, 1, 
          0.62831853071795862 },
  { -0.13857299972299736, 20, 1, 
          0.78539816339744828 },
  { 0.10495324841927710, 20, 1, 
          0.94247779607693793 },
  { -0.075707774352164178, 20, 1, 
          1.0995574287564276 },
  { 0.049168697683476620, 20, 1, 
          1.2566370614359172 },
  { -0.024216050551253254, 20, 1, 
          1.4137166941154069 },
  { 3.9938443510694349e-16, 20, 1, 
          1.5707963267948966 },
  { 0.024216050551250898, 20, 1, 
          1.7278759594743860 },
  { -0.049168697683475482, 20, 1, 
          1.8849555921538759 },
  { 0.075707774352163332, 20, 1, 
          2.0420352248333655 },
  { -0.10495324841927710, 20, 1, 
          2.1991148575128552 },
  { 0.13857299972299736, 20, 1, 
          2.3561944901923448 },
  { -0.17937240823504039, 20, 1, 
          2.5132741228718345 },
  { 0.23278757741246658, 20, 1, 
          2.6703537555513241 },
  { -0.31166370423309170, 20, 1, 
          2.8274333882308138 },
  { 0.45905213045059318, 20, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 20, 1, 
          3.1415926535897931 },
};

// Test function for l=20, m=1.
template <typename Tp>
void test017()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data017)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data017[i].l), Tp(data017[i].m),
                   Tp(data017[i].theta));
      const Tp f0 = data017[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=20, m=2.
testcase_sph_legendre<double> data018[] = {
  { 0.0000000000000000, 20, 2, 
          0.0000000000000000 },
  { 0.87399805141574682, 20, 2, 
          0.15707963267948966 },
  { -0.55116854080894984, 20, 2, 
          0.31415926535897931 },
  { 0.44520137308557534, 20, 2, 
          0.47123889803846897 },
  { -0.39321637877908228, 20, 2, 
          0.62831853071795862 },
  { 0.36312025711350970, 20, 2, 
          0.78539816339744828 },
  { -0.34427103004873094, 20, 2, 
          0.94247779607693793 },
  { 0.33214917638387625, 20, 2, 
          1.0995574287564276 },
  { -0.32455734448839091, 20, 2, 
          1.2566370614359172 },
  { 0.32036529628513238, 20, 2, 
          1.4137166941154069 },
  { -0.31902310563819986, 20, 2, 
          1.5707963267948966 },
  { 0.32036529628513266, 20, 2, 
          1.7278759594743860 },
  { -0.32455734448839102, 20, 2, 
          1.8849555921538759 },
  { 0.33214917638387670, 20, 2, 
          2.0420352248333655 },
  { -0.34427103004873094, 20, 2, 
          2.1991148575128552 },
  { 0.36312025711350970, 20, 2, 
          2.3561944901923448 },
  { -0.39321637877908278, 20, 2, 
          2.5132741228718345 },
  { 0.44520137308557639, 20, 2, 
          2.6703537555513241 },
  { -0.55116854080894984, 20, 2, 
          2.8274333882308138 },
  { 0.87399805141574360, 20, 2, 
          2.9845130209103035 },
  { 0.0000000000000000, 20, 2, 
          3.1415926535897931 },
};

// Test function for l=20, m=2.
template <typename Tp>
void test018()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data018)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data018[i].l), Tp(data018[i].m),
                   Tp(data018[i].theta));
      const Tp f0 = data018[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=20, m=5.
testcase_sph_legendre<double> data019[] = {
  { 0.0000000000000000, 20, 5, 
          0.0000000000000000 },
  { -0.10024848623504846, 20, 5, 
          0.15707963267948966 },
  { -0.68115361075940595, 20, 5, 
          0.31415926535897931 },
  { 0.31774532551156237, 20, 5, 
          0.47123889803846897 },
  { -0.16011868165390564, 20, 5, 
          0.62831853071795862 },
  { 0.085844143304116230, 20, 5, 
          0.78539816339744828 },
  { -0.047467540840864686, 20, 5, 
          0.94247779607693793 },
  { 0.026283575189471282, 20, 5, 
          1.0995574287564276 },
  { -0.013891104052597331, 20, 5, 
          1.2566370614359172 },
  { 0.0059873308239496931, 20, 5, 
          1.4137166941154069 },
  { 3.9355286582083095e-16, 20, 5, 
          1.5707963267948966 },
  { -0.0059873308239519040, 20, 5, 
          1.7278759594743860 },
  { 0.013891104052598531, 20, 5, 
          1.8849555921538759 },
  { -0.026283575189472212, 20, 5, 
          2.0420352248333655 },
  { 0.047467540840864686, 20, 5, 
          2.1991148575128552 },
  { -0.085844143304116230, 20, 5, 
          2.3561944901923448 },
  { 0.16011868165390636, 20, 5, 
          2.5132741228718345 },
  { -0.31774532551156448, 20, 5, 
          2.6703537555513241 },
  { 0.68115361075940595, 20, 5, 
          2.8274333882308138 },
  { 0.10024848623505037, 20, 5, 
          2.9845130209103035 },
  { 0.0000000000000000, 20, 5, 
          3.1415926535897931 },
};

// Test function for l=20, m=5.
template <typename Tp>
void test019()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data019)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data019[i].l), Tp(data019[i].m),
                   Tp(data019[i].theta));
      const Tp f0 = data019[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=20, m=10.
testcase_sph_legendre<double> data020[] = {
  { 0.0000000000000000, 20, 10, 
          0.0000000000000000 },
  { 3.0595797603706485e-05, 20, 10, 
          0.15707963267948966 },
  { 0.015924453916397008, 20, 10, 
          0.31415926535897931 },
  { 0.26588079118745700, 20, 10, 
          0.47123889803846897 },
  { 0.54045081420686825, 20, 10, 
          0.62831853071795862 },
  { -0.28215279394285597, 20, 10, 
          0.78539816339744828 },
  { 0.0085297337582246665, 20, 10, 
          0.94247779607693793 },
  { 0.16930127953533775, 20, 10, 
          1.0995574287564276 },
  { -0.27215134048018352, 20, 10, 
          1.2566370614359172 },
  { 0.32456597088029526, 20, 10, 
          1.4137166941154069 },
  { -0.34057893241353715, 20, 10, 
          1.5707963267948966 },
  { 0.32456597088029449, 20, 10, 
          1.7278759594743860 },
  { -0.27215134048018308, 20, 10, 
          1.8849555921538759 },
  { 0.16930127953533725, 20, 10, 
          2.0420352248333655 },
  { 0.0085297337582246665, 20, 10, 
          2.1991148575128552 },
  { -0.28215279394285597, 20, 10, 
          2.3561944901923448 },
  { 0.54045081420686658, 20, 10, 
          2.5132741228718345 },
  { 0.26588079118745822, 20, 10, 
          2.6703537555513241 },
  { 0.015924453916397008, 20, 10, 
          2.8274333882308138 },
  { 3.0595797603707854e-05, 20, 10, 
          2.9845130209103035 },
  { 0.0000000000000000, 20, 10, 
          3.1415926535897931 },
};

// Test function for l=20, m=10.
template <typename Tp>
void test020()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data020)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data020[i].l), Tp(data020[i].m),
                   Tp(data020[i].theta));
      const Tp f0 = data020[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=20, m=20.
testcase_sph_legendre<double> data021[] = {
  { 0.0000000000000000, 20, 20, 
          0.0000000000000000 },
  { 4.9264471419245886e-17, 20, 20, 
          0.15707963267948966 },
  { 4.0321091681531780e-11, 20, 20, 
          0.31415926535897931 },
  { 8.8474944184471664e-08, 20, 20, 
          0.47123889803846897 },
  { 1.5497395129387764e-05, 20, 20, 
          0.62831853071795862 },
  { 0.00062457564282984723, 20, 20, 
          0.78539816339744828 },
  { 0.0092265192458967603, 20, 20, 
          0.94247779607693793 },
  { 0.063606673236323269, 20, 20, 
          1.0995574287564276 },
  { 0.23442909509776308, 20, 20, 
          1.2566370614359172 },
  { 0.49921030481087009, 20, 20, 
          1.4137166941154069 },
  { 0.63956545825776223, 20, 20, 
          1.5707963267948966 },
  { 0.49921030481087064, 20, 20, 
          1.7278759594743860 },
  { 0.23442909509776344, 20, 20, 
          1.8849555921538759 },
  { 0.063606673236323352, 20, 20, 
          2.0420352248333655 },
  { 0.0092265192458967603, 20, 20, 
          2.1991148575128552 },
  { 0.00062457564282984723, 20, 20, 
          2.3561944901923448 },
  { 1.5497395129387818e-05, 20, 20, 
          2.5132741228718345 },
  { 8.8474944184472617e-08, 20, 20, 
          2.6703537555513241 },
  { 4.0321091681531780e-11, 20, 20, 
          2.8274333882308138 },
  { 4.9264471419250786e-17, 20, 20, 
          2.9845130209103035 },
  { 0.0000000000000000, 20, 20, 
          3.1415926535897931 },
};

// Test function for l=20, m=20.
template <typename Tp>
void test021()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data021)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data021[i].l), Tp(data021[i].m),
                   Tp(data021[i].theta));
      const Tp f0 = data021[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=50, m=0.
testcase_sph_legendre<double> data022[] = {
  { 2.8350175706934717, 50, 0, 
          0.0000000000000000 },
  { 0.53157537495174845, 50, 0, 
          0.15707963267948966 },
  { -0.46056183476301349, 50, 0, 
          0.31415926535897931 },
  { -0.24876032079677909, 50, 0, 
          0.47123889803846897 },
  { 0.36926172901532522, 50, 0, 
          0.62831853071795862 },
  { 0.14571730283563306, 50, 0, 
          0.78539816339744828 },
  { -0.33636199170850811, 50, 0, 
          0.94247779607693793 },
  { -0.079132716267092035, 50, 0, 
          1.0995574287564276 },
  { 0.32232921941301440, 50, 0, 
          1.2566370614359172 },
  { 0.025253991969481446, 50, 0, 
          1.4137166941154069 },
  { -0.31830208724152359, 50, 0, 
          1.5707963267948966 },
  { 0.025253991969476128, 50, 0, 
          1.7278759594743860 },
  { 0.32232921941301479, 50, 0, 
          1.8849555921538759 },
  { -0.079132716267090078, 50, 0, 
          2.0420352248333655 },
  { -0.33636199170850811, 50, 0, 
          2.1991148575128552 },
  { 0.14571730283563306, 50, 0, 
          2.3561944901923448 },
  { 0.36926172901532717, 50, 0, 
          2.5132741228718345 },
  { -0.24876032079677393, 50, 0, 
          2.6703537555513241 },
  { -0.46056183476301349, 50, 0, 
          2.8274333882308138 },
  { 0.53157537495172380, 50, 0, 
          2.9845130209103035 },
  { 2.8350175706934717, 50, 0, 
          3.1415926535897931 },
};

// Test function for l=50, m=0.
template <typename Tp>
void test022()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data022)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data022[i].l), Tp(data022[i].m),
                   Tp(data022[i].theta));
      const Tp f0 = data022[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=50, m=1.
testcase_sph_legendre<double> data023[] = {
  { 0.0000000000000000, 50, 1, 
          0.0000000000000000 },
  { -0.63751752155226116, 50, 1, 
          0.15707963267948966 },
  { -0.32616619317604312, 50, 1, 
          0.31415926535897931 },
  { 0.40649930826162706, 50, 1, 
          0.47123889803846897 },
  { 0.18473991408344026, 50, 1, 
          0.62831853071795862 },
  { -0.35083930302013211, 50, 1, 
          0.78539816339744828 },
  { -0.10755382110947098, 50, 1, 
          0.94247779607693793 },
  { 0.32822568316499862, 50, 1, 
          1.0995574287564276 },
  { 0.050286056609798180, 50, 1, 
          1.2566370614359172 },
  { -0.31935368562159638, 50, 1, 
          1.4137166941154069 },
  { -9.8421602686195941e-16, 50, 1, 
          1.5707963267948966 },
  { 0.31935368562159705, 50, 1, 
          1.7278759594743860 },
  { -0.050286056609795383, 50, 1, 
          1.8849555921538759 },
  { -0.32822568316499923, 50, 1, 
          2.0420352248333655 },
  { 0.10755382110947098, 50, 1, 
          2.1991148575128552 },
  { 0.35083930302013211, 50, 1, 
          2.3561944901923448 },
  { -0.18473991408343632, 50, 1, 
          2.5132741228718345 },
  { -0.40649930826163039, 50, 1, 
          2.6703537555513241 },
  { 0.32616619317604312, 50, 1, 
          2.8274333882308138 },
  { 0.63751752155227837, 50, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 50, 1, 
          3.1415926535897931 },
};

// Test function for l=50, m=1.
template <typename Tp>
void test023()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data023)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data023[i].l), Tp(data023[i].m),
                   Tp(data023[i].theta));
      const Tp f0 = data023[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=50, m=2.
testcase_sph_legendre<double> data024[] = {
  { 0.0000000000000000, 50, 2, 
          0.0000000000000000 },
  { -0.37230261163839168, 50, 2, 
          0.15707963267948966 },
  { 0.50051599680315972, 50, 2, 
          0.31415926535897931 },
  { 0.21724795180329545, 50, 2, 
          0.47123889803846897 },
  { -0.37948127307610940, 50, 2, 
          0.62831853071795862 },
  { -0.13187372121003119, 50, 2, 
          0.78539816339744828 },
  { 0.33959009162400194, 50, 2, 
          0.94247779607693793 },
  { 0.072537503112490409, 50, 2, 
          1.0995574287564276 },
  { -0.32310306941855271, 50, 2, 
          1.2566370614359172 },
  { -0.023259822816436588, 50, 2, 
          1.4137166941154069 },
  { 0.31842698506357275, 50, 2, 
          1.5707963267948966 },
  { -0.023259822816431144, 50, 2, 
          1.7278759594743860 },
  { -0.32310306941855299, 50, 2, 
          1.8849555921538759 },
  { 0.072537503112488369, 50, 2, 
          2.0420352248333655 },
  { 0.33959009162400194, 50, 2, 
          2.1991148575128552 },
  { -0.13187372121003119, 50, 2, 
          2.3561944901923448 },
  { -0.37948127307611074, 50, 2, 
          2.5132741228718345 },
  { 0.21724795180328935, 50, 2, 
          2.6703537555513241 },
  { 0.50051599680315972, 50, 2, 
          2.8274333882308138 },
  { -0.37230261163836298, 50, 2, 
          2.9845130209103035 },
  { 0.0000000000000000, 50, 2, 
          3.1415926535897931 },
};

// Test function for l=50, m=2.
template <typename Tp>
void test024()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data024)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data024[i].l), Tp(data024[i].m),
                   Tp(data024[i].theta));
      const Tp f0 = data024[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000015e-12));
}

// Test data for l=50, m=5.
testcase_sph_legendre<double> data025[] = {
  { 0.0000000000000000, 50, 5, 
          0.0000000000000000 },
  { -0.57750385903193069, 50, 5, 
          0.15707963267948966 },
  { 0.077360497065584566, 50, 5, 
          0.31415926535897931 },
  { 0.47707267400540210, 50, 5, 
          0.47123889803846897 },
  { 0.055370615126630537, 50, 5, 
          0.62831853071795862 },
  { -0.37629451847202855, 50, 5, 
          0.78539816339744828 },
  { -0.048042277801960624, 50, 5, 
          0.94247779607693793 },
  { 0.33619379362228685, 50, 5, 
          1.0995574287564276 },
  { 0.025265227185719726, 50, 5, 
          1.2566370614359172 },
  { -0.32083679430964518, 50, 5, 
          1.4137166941154069 },
  { -9.8189201019751884e-16, 50, 5, 
          1.5707963267948966 },
  { 0.32083679430964590, 50, 5, 
          1.7278759594743860 },
  { -0.025265227185716856, 50, 5, 
          1.8849555921538759 },
  { -0.33619379362228730, 50, 5, 
          2.0420352248333655 },
  { 0.048042277801960624, 50, 5, 
          2.1991148575128552 },
  { 0.37629451847202855, 50, 5, 
          2.3561944901923448 },
  { -0.055370615126626936, 50, 5, 
          2.5132741228718345 },
  { -0.47707267400540210, 50, 5, 
          2.6703537555513241 },
  { -0.077360497065584566, 50, 5, 
          2.8274333882308138 },
  { 0.57750385903191004, 50, 5, 
          2.9845130209103035 },
  { 0.0000000000000000, 50, 5, 
          3.1415926535897931 },
};

// Test function for l=50, m=5.
template <typename Tp>
void test025()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data025)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data025[i].l), Tp(data025[i].m),
                   Tp(data025[i].theta));
      const Tp f0 = data025[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=50, m=10.
testcase_sph_legendre<double> data026[] = {
  { 0.0000000000000000, 50, 10, 
          0.0000000000000000 },
  { 0.15606941844800759, 50, 10, 
          0.15707963267948966 },
  { -0.53748868836814601, 50, 10, 
          0.31415926535897931 },
  { -0.49304919025183896, 50, 10, 
          0.47123889803846897 },
  { -0.26267582750428264, 50, 10, 
          0.62831853071795862 },
  { 0.22058983666314402, 50, 10, 
          0.78539816339744828 },
  { 0.32936725160671759, 50, 10, 
          0.94247779607693793 },
  { -0.092053311559446988, 50, 10, 
          1.0995574287564276 },
  { -0.32542913495935555, 50, 10, 
          1.2566370614359172 },
  { 0.025673223789103500, 50, 10, 
          1.4137166941154069 },
  { 0.32150019350255743, 50, 10, 
          1.5707963267948966 },
  { 0.025673223789108864, 50, 10, 
          1.7278759594743860 },
  { -0.32542913495935494, 50, 10, 
          1.8849555921538759 },
  { -0.092053311559448570, 50, 10, 
          2.0420352248333655 },
  { 0.32936725160671759, 50, 10, 
          2.1991148575128552 },
  { 0.22058983666314402, 50, 10, 
          2.3561944901923448 },
  { -0.26267582750427909, 50, 10, 
          2.5132741228718345 },
  { -0.49304919025184119, 50, 10, 
          2.6703537555513241 },
  { -0.53748868836814601, 50, 10, 
          2.8274333882308138 },
  { 0.15606941844801259, 50, 10, 
          2.9845130209103035 },
  { 0.0000000000000000, 50, 10, 
          3.1415926535897931 },
};

// Test function for l=50, m=10.
template <typename Tp>
void test026()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data026)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data026[i].l), Tp(data026[i].m),
                   Tp(data026[i].theta));
      const Tp f0 = data026[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=50, m=20.
testcase_sph_legendre<double> data027[] = {
  { 0.0000000000000000, 50, 20, 
          0.0000000000000000 },
  { 3.0409598712833082e-07, 50, 20, 
          0.15707963267948966 },
  { 0.030940518122882305, 50, 20, 
          0.31415926535897931 },
  { 0.64134588721659935, 50, 20, 
          0.47123889803846897 },
  { 0.29895244392136394, 50, 20, 
          0.62831853071795862 },
  { 0.25309324781874065, 50, 20, 
          0.78539816339744828 },
  { 0.34368634714931712, 50, 20, 
          0.94247779607693793 },
  { 0.33996764360663956, 50, 20, 
          1.0995574287564276 },
  { 0.12866267745104118, 50, 20, 
          1.2566370614359172 },
  { -0.18201114398922874, 50, 20, 
          1.4137166941154069 },
  { -0.33216683431510857, 50, 20, 
          1.5707963267948966 },
  { -0.18201114398923304, 50, 20, 
          1.7278759594743860 },
  { 0.12866267745103846, 50, 20, 
          1.8849555921538759 },
  { 0.33996764360663906, 50, 20, 
          2.0420352248333655 },
  { 0.34368634714931712, 50, 20, 
          2.1991148575128552 },
  { 0.25309324781874065, 50, 20, 
          2.3561944901923448 },
  { 0.29895244392136738, 50, 20, 
          2.5132741228718345 },
  { 0.64134588721659791, 50, 20, 
          2.6703537555513241 },
  { 0.030940518122882305, 50, 20, 
          2.8274333882308138 },
  { 3.0409598712835877e-07, 50, 20, 
          2.9845130209103035 },
  { 0.0000000000000000, 50, 20, 
          3.1415926535897931 },
};

// Test function for l=50, m=20.
template <typename Tp>
void test027()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data027)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data027[i].l), Tp(data027[i].m),
                   Tp(data027[i].theta));
      const Tp f0 = data027[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=50, m=50.
testcase_sph_legendre<double> data028[] = {
  { 0.0000000000000000, 50, 50, 
          0.0000000000000000 },
  { 4.1649039898151844e-41, 50, 50, 
          0.15707963267948966 },
  { 2.5240684647724192e-26, 50, 50, 
          0.31415926535897931 },
  { 5.6927376423967334e-18, 50, 50, 
          0.47123889803846897 },
  { 2.3116239814797057e-12, 50, 50, 
          0.62831853071795862 },
  { 2.3835981241325311e-08, 50, 50, 
          0.78539816339744828 },
  { 1.9992410287270356e-05, 50, 50, 
          0.94247779607693793 },
  { 0.0024947505670829791, 50, 50, 
          1.0995574287564276 },
  { 0.065057774647971175, 50, 50, 
          1.2566370614359172 },
  { 0.43050607056732243, 50, 50, 
          1.4137166941154069 },
  { 0.79980281171531975, 50, 50, 
          1.5707963267948966 },
  { 0.43050607056732360, 50, 50, 
          1.7278759594743860 },
  { 0.065057774647971384, 50, 50, 
          1.8849555921538759 },
  { 0.0024947505670829856, 50, 50, 
          2.0420352248333655 },
  { 1.9992410287270356e-05, 50, 50, 
          2.1991148575128552 },
  { 2.3835981241325311e-08, 50, 50, 
          2.3561944901923448 },
  { 2.3116239814797222e-12, 50, 50, 
          2.5132741228718345 },
  { 5.6927376423968544e-18, 50, 50, 
          2.6703537555513241 },
  { 2.5240684647724192e-26, 50, 50, 
          2.8274333882308138 },
  { 4.1649039898161316e-41, 50, 50, 
          2.9845130209103035 },
  { 0.0000000000000000, 50, 50, 
          3.1415926535897931 },
};

// Test function for l=50, m=50.
template <typename Tp>
void test028()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data028)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data028[i].l), Tp(data028[i].m),
                   Tp(data028[i].theta));
      const Tp f0 = data028[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=100, m=0.
testcase_sph_legendre<double> data029[] = {
  { 3.9993839251484076, 100, 0, 
          0.0000000000000000 },
  { -0.60770160285935471, 100, 0, 
          0.15707963267948966 },
  { 0.46193027883956100, 100, 0, 
          0.31415926535897931 },
  { -0.40218718869815234, 100, 0, 
          0.47123889803846897 },
  { 0.36960201406910737, 100, 0, 
          0.62831853071795862 },
  { -0.34953726547378611, 100, 0, 
          0.78539816339744828 },
  { 0.33646959352497846, 100, 0, 
          0.94247779607693793 },
  { -0.32784733067663169, 100, 0, 
          1.0995574287564276 },
  { 0.32235624474047936, 100, 0, 
          1.2566370614359172 },
  { -0.31929330706601283, 100, 0, 
          1.4137166941154069 },
  { 0.31830791662110325, 100, 0, 
          1.5707963267948966 },
  { -0.31929330706601389, 100, 0, 
          1.7278759594743860 },
  { 0.32235624474048052, 100, 0, 
          1.8849555921538759 },
  { -0.32784733067663291, 100, 0, 
          2.0420352248333655 },
  { 0.33646959352497846, 100, 0, 
          2.1991148575128552 },
  { -0.34953726547378611, 100, 0, 
          2.3561944901923448 },
  { 0.36960201406911114, 100, 0, 
          2.5132741228718345 },
  { -0.40218718869815695, 100, 0, 
          2.6703537555513241 },
  { 0.46193027883956100, 100, 0, 
          2.8274333882308138 },
  { -0.60770160285939478, 100, 0, 
          2.9845130209103035 },
  { 3.9993839251484076, 100, 0, 
          3.1415926535897931 },
};

// Test function for l=100, m=0.
template <typename Tp>
void test029()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data029)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data029[i].l), Tp(data029[i].m),
                   Tp(data029[i].theta));
      const Tp f0 = data029[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=100, m=1.
testcase_sph_legendre<double> data030[] = {
  { 0.0000000000000000, 100, 1, 
          0.0000000000000000 },
  { -0.50851949013719622, 100, 1, 
          0.15707963267948966 },
  { 0.33129641402221310, 100, 1, 
          0.31415926535897931 },
  { -0.24390405750942562, 100, 1, 
          0.47123889803846897 },
  { 0.18659755088414165, 100, 1, 
          0.62831853071795862 },
  { -0.14355908970516640, 100, 1, 
          0.78539816339744828 },
  { 0.10844906813251093, 100, 1, 
          0.94247779607693793 },
  { -0.078100088690859812, 100, 1, 
          1.0995574287564276 },
  { 0.050670002998304528, 100, 1, 
          1.2566370614359172 },
  { -0.024941251747138762, 100, 1, 
          1.4137166941154069 },
  { 1.9587949830851639e-15, 100, 1, 
          1.5707963267948966 },
  { 0.024941251747127649, 100, 1, 
          1.7278759594743860 },
  { -0.050670002998298595, 100, 1, 
          1.8849555921538759 },
  { 0.078100088690855676, 100, 1, 
          2.0420352248333655 },
  { -0.10844906813251093, 100, 1, 
          2.1991148575128552 },
  { 0.14355908970516640, 100, 1, 
          2.3561944901923448 },
  { -0.18659755088413349, 100, 1, 
          2.5132741228718345 },
  { 0.24390405750941485, 100, 1, 
          2.6703537555513241 },
  { -0.33129641402221310, 100, 1, 
          2.8274333882308138 },
  { 0.50851949013714548, 100, 1, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 1, 
          3.1415926535897931 },
};

// Test function for l=100, m=1.
template <typename Tp>
void test030()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data030)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data030[i].l), Tp(data030[i].m),
                   Tp(data030[i].theta));
      const Tp f0 = data030[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=100, m=2.
testcase_sph_legendre<double> data031[] = {
  { 0.0000000000000000, 100, 2, 
          0.0000000000000000 },
  { 0.67166274297194040, 100, 2, 
          0.15707963267948966 },
  { -0.48226933687995144, 100, 2, 
          0.31415926535897931 },
  { 0.41175421895715447, 100, 2, 
          0.47123889803846897 },
  { -0.37475021787822460, 100, 2, 
          0.62831853071795862 },
  { 0.35242909383605475, 100, 2, 
          0.78539816339744828 },
  { -0.33807110409160002, 100, 2, 
          0.94247779607693793 },
  { 0.32867180390709999, 100, 2, 
          1.0995574287564276 },
  { -0.32271583790278469, 100, 2, 
          1.2566370614359172 },
  { 0.31940354677687433, 100, 2, 
          1.4137166941154069 },
  { -0.31833943693772526, 100, 2, 
          1.5707963267948966 },
  { 0.31940354677687521, 100, 2, 
          1.7278759594743860 },
  { -0.32271583790278524, 100, 2, 
          1.8849555921538759 },
  { 0.32867180390710143, 100, 2, 
          2.0420352248333655 },
  { -0.33807110409160002, 100, 2, 
          2.1991148575128552 },
  { 0.35242909383605475, 100, 2, 
          2.3561944901923448 },
  { -0.37475021787822771, 100, 2, 
          2.5132741228718345 },
  { 0.41175421895716069, 100, 2, 
          2.6703537555513241 },
  { -0.48226933687995144, 100, 2, 
          2.8274333882308138 },
  { 0.67166274297196804, 100, 2, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 2, 
          3.1415926535897931 },
};

// Test function for l=100, m=2.
template <typename Tp>
void test031()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data031)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data031[i].l), Tp(data031[i].m),
                   Tp(data031[i].theta));
      const Tp f0 = data031[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for l=100, m=5.
testcase_sph_legendre<double> data032[] = {
  { 0.0000000000000000, 100, 5, 
          0.0000000000000000 },
  { 0.062564361105907143, 100, 5, 
          0.15707963267948966 },
  { 0.14179554455880186, 100, 5, 
          0.31415926535897931 },
  { -0.14356866942906019, 100, 5, 
          0.47123889803846897 },
  { 0.12355483388448507, 100, 5, 
          0.62831853071795862 },
  { -0.10090029999681098, 100, 5, 
          0.78539816339744828 },
  { 0.078905134460230564, 100, 5, 
          0.94247779607693793 },
  { -0.058040182398187236, 100, 5, 
          1.0995574287564276 },
  { 0.038142759389484152, 100, 5, 
          1.2566370614359172 },
  { -0.018906264170660277, 100, 5, 
          1.4137166941154069 },
  { 1.9576303042914544e-15, 100, 5, 
          1.5707963267948966 },
  { 0.018906264170649747, 100, 5, 
          1.7278759594743860 },
  { -0.038142759389478524, 100, 5, 
          1.8849555921538759 },
  { 0.058040182398182996, 100, 5, 
          2.0420352248333655 },
  { -0.078905134460230564, 100, 5, 
          2.1991148575128552 },
  { 0.10090029999681098, 100, 5, 
          2.3561944901923448 },
  { -0.12355483388447780, 100, 5, 
          2.5132741228718345 },
  { 0.14356866942904903, 100, 5, 
          2.6703537555513241 },
  { -0.14179554455880186, 100, 5, 
          2.8274333882308138 },
  { -0.062564361105959004, 100, 5, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 5, 
          3.1415926535897931 },
};

// Test function for l=100, m=5.
template <typename Tp>
void test032()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data032)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data032[i].l), Tp(data032[i].m),
                   Tp(data032[i].theta));
      const Tp f0 = data032[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000029e-12));
}

// Test data for l=100, m=10.
testcase_sph_legendre<double> data033[] = {
  { 0.0000000000000000, 100, 10, 
          0.0000000000000000 },
  { -0.75366545187995670, 100, 10, 
          0.15707963267948966 },
  { -0.35914570017277186, 100, 10, 
          0.31415926535897931 },
  { 0.43480692911578245, 100, 10, 
          0.47123889803846897 },
  { -0.40862111080315705, 100, 10, 
          0.62831853071795862 },
  { 0.37832688692909400, 100, 10, 
          0.78539816339744828 },
  { -0.35484056194773472, 100, 10, 
          0.94247779607693793 },
  { 0.33821981171196336, 100, 10, 
          1.0995574287564276 },
  { -0.32729120767830605, 100, 10, 
          1.2566370614359172 },
  { 0.32110336937091455, 100, 10, 
          1.4137166941154069 },
  { -0.31910064020036194, 100, 10, 
          1.5707963267948966 },
  { 0.32110336937091488, 100, 10, 
          1.7278759594743860 },
  { -0.32729120767830577, 100, 10, 
          1.8849555921538759 },
  { 0.33821981171196341, 100, 10, 
          2.0420352248333655 },
  { -0.35484056194773472, 100, 10, 
          2.1991148575128552 },
  { 0.37832688692909400, 100, 10, 
          2.3561944901923448 },
  { -0.40862111080315433, 100, 10, 
          2.5132741228718345 },
  { 0.43480692911577806, 100, 10, 
          2.6703537555513241 },
  { -0.35914570017277186, 100, 10, 
          2.8274333882308138 },
  { -0.75366545187998180, 100, 10, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 10, 
          3.1415926535897931 },
};

// Test function for l=100, m=10.
template <typename Tp>
void test033()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data033)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data033[i].l), Tp(data033[i].m),
                   Tp(data033[i].theta));
      const Tp f0 = data033[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for l=100, m=20.
testcase_sph_legendre<double> data034[] = {
  { 0.0000000000000000, 100, 20, 
          0.0000000000000000 },
  { 0.053569660841553138, 100, 20, 
          0.15707963267948966 },
  { 0.57154926874732426, 100, 20, 
          0.31415926535897931 },
  { 0.47536909969585828, 100, 20, 
          0.47123889803846897 },
  { -0.28882554564109014, 100, 20, 
          0.62831853071795862 },
  { 0.020116179014049645, 100, 20, 
          0.78539816339744828 },
  { 0.14752195931706580, 100, 20, 
          0.94247779607693793 },
  { -0.24069428588868527, 100, 20, 
          1.0995574287564276 },
  { 0.29031796025014395, 100, 20, 
          1.2566370614359172 },
  { -0.31437256851143475, 100, 20, 
          1.4137166941154069 },
  { 0.32153954851141792, 100, 20, 
          1.5707963267948966 },
  { -0.31437256851143192, 100, 20, 
          1.7278759594743860 },
  { 0.29031796025014112, 100, 20, 
          1.8849555921538759 },
  { -0.24069428588868211, 100, 20, 
          2.0420352248333655 },
  { 0.14752195931706580, 100, 20, 
          2.1991148575128552 },
  { 0.020116179014049645, 100, 20, 
          2.3561944901923448 },
  { -0.28882554564109658, 100, 20, 
          2.5132741228718345 },
  { 0.47536909969585378, 100, 20, 
          2.6703537555513241 },
  { 0.57154926874732426, 100, 20, 
          2.8274333882308138 },
  { 0.053569660841557065, 100, 20, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 20, 
          3.1415926535897931 },
};

// Test function for l=100, m=20.
template <typename Tp>
void test034()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data034)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data034[i].l), Tp(data034[i].m),
                   Tp(data034[i].theta));
      const Tp f0 = data034[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000015e-12));
}

// Test data for l=100, m=50.
testcase_sph_legendre<double> data035[] = {
  { 0.0000000000000000, 100, 50, 
          0.0000000000000000 },
  { 3.3047910392590630e-21, 100, 50, 
          0.15707963267948966 },
  { 1.0592655372554983e-07, 100, 50, 
          0.31415926535897931 },
  { 0.080418744223952635, 100, 50, 
          0.47123889803846897 },
  { -0.56450600580393062, 100, 50, 
          0.62831853071795862 },
  { 0.33338739844741766, 100, 50, 
          0.78539816339744828 },
  { 0.39741714816514678, 100, 50, 
          0.94247779607693793 },
  { 0.35223993750972293, 100, 50, 
          1.0995574287564276 },
  { 0.17885891940721749, 100, 50, 
          1.2566370614359172 },
  { -0.15341660126461953, 100, 50, 
          1.4137166941154069 },
  { -0.34175924303503102, 100, 50, 
          1.5707963267948966 },
  { -0.15341660126462864, 100, 50, 
          1.7278759594743860 },
  { 0.17885891940721332, 100, 50, 
          1.8849555921538759 },
  { 0.35223993750972149, 100, 50, 
          2.0420352248333655 },
  { 0.39741714816514678, 100, 50, 
          2.1991148575128552 },
  { 0.33338739844741766, 100, 50, 
          2.3561944901923448 },
  { -0.56450600580392785, 100, 50, 
          2.5132741228718345 },
  { 0.080418744223953439, 100, 50, 
          2.6703537555513241 },
  { 1.0592655372554983e-07, 100, 50, 
          2.8274333882308138 },
  { 3.3047910392597871e-21, 100, 50, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 50, 
          3.1415926535897931 },
};

// Test function for l=100, m=50.
template <typename Tp>
void test035()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data035)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data035[i].l), Tp(data035[i].m),
                   Tp(data035[i].theta));
      const Tp f0 = data035[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for l=100, m=100.
testcase_sph_legendre<double> data036[] = {
  { 0.0000000000000000, 100, 100, 
          0.0000000000000000 },
  { 2.5744136608862186e-81, 100, 100, 
          0.15707963267948966 },
  { 9.4551974868956498e-52, 100, 100, 
          0.31415926535897931 },
  { 4.8096190703396912e-35, 100, 100, 
          0.47123889803846897 },
  { 7.9305393636343450e-24, 100, 100, 
          0.62831853071795862 },
  { 8.4320740610946652e-16, 100, 100, 
          0.78539816339744828 },
  { 5.9319660146027522e-10, 100, 100, 
          0.94247779607693793 },
  { 9.2368225946796921e-06, 100, 100, 
          1.0995574287564276 },
  { 0.0062815489742043982, 100, 100, 
          1.2566370614359172 },
  { 0.27505966018176986, 100, 100, 
          1.4137166941154069 },
  { 0.94936713998764621, 100, 100, 
          1.5707963267948966 },
  { 0.27505966018177136, 100, 100, 
          1.7278759594743860 },
  { 0.0062815489742044433, 100, 100, 
          1.8849555921538759 },
  { 9.2368225946797582e-06, 100, 100, 
          2.0420352248333655 },
  { 5.9319660146027522e-10, 100, 100, 
          2.1991148575128552 },
  { 8.4320740610946652e-16, 100, 100, 
          2.3561944901923448 },
  { 7.9305393636344581e-24, 100, 100, 
          2.5132741228718345 },
  { 4.8096190703399648e-35, 100, 100, 
          2.6703537555513241 },
  { 9.4551974868956498e-52, 100, 100, 
          2.8274333882308138 },
  { 2.5744136608873895e-81, 100, 100, 
          2.9845130209103035 },
  { 0.0000000000000000, 100, 100, 
          3.1415926535897931 },
};

// Test function for l=100, m=100.
template <typename Tp>
void test036()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data036)
                         / sizeof(testcase_sph_legendre<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::sph_legendre(Tp(data036[i].l), Tp(data036[i].m),
                   Tp(data036[i].theta));
      const Tp f0 = data036[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps
       && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

int main(int, char**)
{
  test001<double>();
  test002<double>();
  test003<double>();
  test004<double>();
  test005<double>();
  test006<double>();
  test007<double>();
  test008<double>();
  test009<double>();
  test010<double>();
  test011<double>();
  test012<double>();
  test013<double>();
  test014<double>();
  test015<double>();
  test016<double>();
  test017<double>();
  test018<double>();
  test019<double>();
  test020<double>();
  test021<double>();
  test022<double>();
  test023<double>();
  test024<double>();
  test025<double>();
  test026<double>();
  test027<double>();
  test028<double>();
  test029<double>();
  test030<double>();
  test031<double>();
  test032<double>();
  test033<double>();
  test034<double>();
  test035<double>();
  test036<double>();
  return 0;
}
