/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var isnan = require( './../../../../base/assert/is-nan' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var EPS = require( '@stdlib/constants/float64/eps' );
var abs = require( './../../../../base/special/abs' );
var pow = require( './../../../../base/special/pow' );
var sincospi = require( './../lib/' );


// FIXTURES //

var integers = require( './fixtures/julia/integers.json' );
var decimals = require( './fixtures/julia/decimals.json' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof sincospi, 'function', 'main export is a function' );
	t.end();
});

tape( 'if provided negative infinity, the function returns `[NaN,NaN]`', function test( t ) {
	var y = sincospi( NINF );
	t.equal( isnan( y[ 0 ] ), true, 'returns NaN when provided negative infinity' );
	t.equal( isnan( y[ 1 ] ), true, 'returns NaN when provided negative infinity' );
	t.end();
});

tape( 'if provided positive infinity, the function returns `[NaN,NaN]`', function test( t ) {
	var y = sincospi( PINF );
	t.equal( isnan( y[ 0 ] ), true, 'returns NaN when provided positive infinity' );
	t.equal( isnan( y[ 1 ] ), true, 'returns NaN when provided positive infinity' );
	t.end();
});

tape( 'if provided `NaN`, the function returns `[NaN,NaN]`', function test( t ) {
	var y = sincospi( NaN );
	t.equal( isnan( y[ 0 ] ), true, 'returns NaN when provided NaN' );
	t.equal( isnan( y[ 1 ] ), true, 'returns NaN when provided NaN' );
	t.end();
});

tape( 'the function computes results in-place and returns a reference to the input array', function test( t ) {
	var out = [ 0.0, 0.0 ];
	var y = sincospi( out, 0.0 );
	t.deepEqual( y, [ 0.0, 1.0 ], 'computes the correct result' );
	t.equal( y, out, 'returns a reference to the input array' );
	t.end();
});

tape( 'the function computes sin(πx) and cos(πx) for integer input', function test( t ) {
	var c;
	var s;
	var x;
	var y;
	var i;

	x = integers.x;
	s = integers.sin;
	c = integers.cos;
	for ( i = 0; i < x.length; i++ ) {
		y = sincospi( x[ i ] );
		t.equal( y[ 0 ], s[ i ], 'returns '+s[ i ] );
		t.equal( y[ 1 ], c[ i ], 'returns '+c[ i ] );
	}
	t.end();
});

tape( 'if provided a value exceeding `2**53` (max (unsafe) float64 integer), the function returns [0,1]', function test( t ) {
	var x;
	var y;
	var i;

	x = pow( 2.0, 53 ) + 1.0;
	for ( i = 0; i < 100; i++ ) {
		y = sincospi( x+i );
		t.equal( y[ 0 ], 0.0, 'returns 0.0' );
		t.equal( y[ 1 ], 1.0, 'returns 1.0' );
	}
	t.end();
});

tape( 'the function computes the sin(πx) and cos(πx) for fractional part equal to 1/2', function test( t ) {
	var x;
	var y;
	var i;

	for ( i = -100; i <= 100; i++ ) {
		x = 0.5 + ( 1.0 * i );
		y = sincospi( x );
		t.equal( y[ 0 ], ( (i%2 === 0) ? 1.0 : -1.0 ), 'x: '+x+'. Expected: 0' );
		t.equal( y[ 1 ], 0.0, 'x: '+x+'. Expected: 0' );
	}
	t.end();
});

tape( 'the function computes the sin(πx) and cos(πx) for decimal input', function test( t ) {
	var delta;
	var x;
	var y;
	var i;
	var s;
	var c;

	x = decimals.x;
	s = decimals.sin;
	c = decimals.cos;
	for ( i = 0; i < x.length; i++ ) {
		y = sincospi( x[ i ] );

		if ( y[ 0 ] === s[ i ] ) {
			t.equal( y[ 0 ], s[ i ], 'x: '+x[ i ]+'. Expected: '+s[ i ] );
		} else {
			delta = abs( y[ 0 ] - s[ i ] );
			t.ok( delta <= EPS, 'within tolerance. x: '+x[ i ]+'. Value: '+y[ 0 ]+'. Expected: '+s[ i ]+'. Tolerance: '+EPS+'.' );
		}
		if ( y[ 1 ] === c[ i ] ) {
			t.equal( y[ 1 ], c[ i ], 'x: '+x[ i ]+'. Expected: '+c[ i ] );
		} else {
			delta = abs( y[ 1 ] - c[ i ] );
			t.ok( delta <= EPS, 'within tolerance. x: '+x[ i ]+'. Value: '+y[ 1 ]+'. Expected: '+c[ i ]+'. Tolerance: '+EPS+'.' );
		}
	}
	t.end();
});
