// -*- mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// vi: set et ts=4 sw=4 sts=4:
/*
  This file is part of the Open Porous Media project (OPM).

  OPM is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 2 of the License, or
  (at your option) any later version.

  OPM is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OPM.  If not, see <http://www.gnu.org/licenses/>.

  Consult the COPYING file in the top-level source directory of this
  module for the precise wording of the license and the list of
  copyright holders.
*/

#include <config.h>
#include <opm/material/fluidsystems/blackoilpvt/ConstantCompressibilityOilPvt.hpp>

#include <opm/common/ErrorMacros.hpp>

#include <opm/input/eclipse/EclipseState/EclipseState.hpp>

#include <fmt/format.h>

namespace Opm {

#if HAVE_ECL_INPUT
template<class Scalar>
void ConstantCompressibilityOilPvt<Scalar>::
initFromState(const EclipseState& eclState, const Schedule&)
{
    const auto& pvcdoTable = eclState.getTableManager().getPvcdoTable();
    const auto& densityTable = eclState.getTableManager().getDensityTable();

    if (pvcdoTable.size() != densityTable.size()) {
        OPM_THROW(std::runtime_error,
                  fmt::format("Table sizes mismatch. PVCDO: {}, DensityTable: {}\n",
                              pvcdoTable.size(), densityTable.size()));
    }

    std::size_t regions = pvcdoTable.size();
    setNumRegions(regions);

    for (unsigned regionIdx = 0; regionIdx < regions; ++regionIdx) {
        Scalar rhoRefO = densityTable[regionIdx].oil;
        Scalar rhoRefG = densityTable[regionIdx].gas;
        Scalar rhoRefW = densityTable[regionIdx].water;

        setReferenceDensities(regionIdx, rhoRefO, rhoRefG, rhoRefW);

        oilReferencePressure_[regionIdx] = pvcdoTable[regionIdx].reference_pressure;
        oilReferenceFormationVolumeFactor_[regionIdx] = pvcdoTable[regionIdx].volume_factor;
        oilCompressibility_[regionIdx] = pvcdoTable[regionIdx].compressibility;
        oilViscosity_[regionIdx] = pvcdoTable[regionIdx].viscosity;
        oilViscosibility_[regionIdx] = pvcdoTable[regionIdx].viscosibility;
    }

    initEnd();
}
#endif

template<class Scalar>
void ConstantCompressibilityOilPvt<Scalar>::
setNumRegions(std::size_t numRegions)
{
    oilReferenceDensity_.resize(numRegions);
    oilReferencePressure_.resize(numRegions);
    oilReferenceFormationVolumeFactor_.resize(numRegions);
    oilCompressibility_.resize(numRegions);
    oilViscosity_.resize(numRegions);
    oilViscosibility_.resize(numRegions);

    for (unsigned regionIdx = 0; regionIdx < numRegions; ++regionIdx) {
        setReferenceFormationVolumeFactor(regionIdx, 1.0);
        setReferencePressure(regionIdx, 1.03125);
    }
}

template class ConstantCompressibilityOilPvt<double>;
template class ConstantCompressibilityOilPvt<float>;

} // namespace Opm
