% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/layers.R
\name{sass_layer}
\alias{sass_layer}
\alias{sass_layer_file}
\alias{sass_bundle}
\alias{sass_bundle_remove}
\alias{is_sass_bundle}
\title{Bundling Sass layers}
\usage{
sass_layer(
  functions = NULL,
  defaults = NULL,
  mixins = NULL,
  rules = NULL,
  html_deps = NULL,
  file_attachments = character(0),
  declarations = NULL,
  tags = NULL
)

sass_layer_file(file)

sass_bundle(...)

sass_bundle_remove(bundle, name)

is_sass_bundle(x)
}
\arguments{
\item{functions}{\code{\link[=as_sass]{as_sass()}} \code{input} intended for \href{https://rstudio.github.io/sass/articles/sass.html#functions-1}{Sass functions}.
Functions are placed before \code{defaults} so that variable definitions may make
use of functions.}

\item{defaults}{\code{\link[=as_sass]{as_sass()}} \code{input} intended for \href{https://rstudio.github.io/sass/articles/sass.html#variables-1}{variable defaults}.
These variable defaults after placed after \code{functions} but before \code{mixins}.
When multiple layers are combined in a \code{sass_bundle()}, defaults are merged
in reverse order; that is, \code{sass_bundle(layer1, layer2)} will include
\code{layer2$defaults} before \code{layer1$defaults}.}

\item{mixins}{\code{\link[=as_sass]{as_sass()}} \code{input} intended for \href{https://rstudio.github.io/sass/articles/sass.html#mixins-1}{Sass mixins}. Mixins
are placed after \code{defaults}, but before \code{rules}.}

\item{rules}{\code{\link[=as_sass]{as_sass()}} \code{input} intended for \href{https://sass-lang.com/documentation/style-rules}{Sass rules}. Rules are placed last
(i.e., after \code{functions}, \code{defaults}, and \code{mixins}).}

\item{html_deps}{An HTML dependency (or a list of them). This dependency
gets attached to the return value of \code{\link[=sass]{sass()}}/\code{\link[=as_sass]{as_sass()}}.}

\item{file_attachments}{A named character vector, representing file assets
that are referenced (using relative paths) from the sass in this layer. The
vector names should be a relative path, and the corresponding vector values
should be absolute paths to files or directories that exist; at render
time, each value will be copied to the relative path indicated by its name.
(For directories, the \emph{contents} of the source directory will be copied
into the destination directory; the directory itself will not be copied.)
You can also omit the name, in which case that file or directory will be
copied directly into the output directory.}

\item{declarations}{Deprecated, use \code{functions} or \code{mixins} instead.}

\item{tags}{Deprecated. Preserve meta information using a key in \code{sass_bundle(KEY = val)}.
preserve simple metadata as layers are merged.}

\item{file}{file path to a \code{.scss} file.}

\item{...}{A collection of \code{sass_layer()}s and/or objects that \code{\link[=as_sass]{as_sass()}}
understands. Arguments should be provided in reverse priority order:
defaults, declarations, and rules in later layers will take precedence over
those of previous layers. Non-layer values will be converted to layers by
calling \code{sass_layer(rules = ...)}.}

\item{bundle}{Output value from \code{sass_layer()} or \code{sass_bundle()}}

\item{name}{If a Sass layer name is contained in \code{name}, the matching Sass layer will be removed from the \code{bundle}}

\item{x}{object to inspect}
}
\description{
Sass layers provide a way to package Sass variables, rules, functions, and
mixins in a structured and composable way that follows best Sass practices.
Most importantly, when multiple \code{sass_layer()} are combined into a
\code{sass_bundle()}, variable \code{defaults} for later layers are placed \emph{before}
earlier layers, effectively 'new' defaults through all the 'old' defaults.
}
\section{Functions}{
\itemize{
\item \code{sass_layer}: Compose the parts of a single Sass layer. Object returned is a \code{sass_bundle()} with a single Sass layer

\item \code{sass_layer_file}: Read in a \code{.scss} file with parse special \verb{/*-- scss:(functions|defaults|rules|mixins) --*/} comments as relevant sections of a \code{sass_layer()}.

\item \code{sass_bundle}: Collect \code{sass_bundle()} and/or \code{sass_layer()} objects. Unnamed Sass bundles will be concatenated together, preserving their internal name structures. Named Sass bundles will be condensed into a single Sass layer for easier removal from the returned Sass bundle.

\item \code{sass_bundle_remove}: Remove a whole \code{sass_layer()} from a \code{sass_bundle()} object.

\item \code{is_sass_bundle}: Check if \code{x} is a Sass bundle object
}}

\examples{
blue <- list(color = "blue !default")
red <- list(color = "red !default")
green <- list(color = "green !default")

# a sass_layer() by itself is not very useful, it just defines some
# SASS to place before (defaults) and after (rules)
core <- sass_layer(defaults = blue, rules = "body { color: $color; }")
core
sass(core)

# However, by stacking sass_layer()s, we have ability to place
# SASS both before and after some other sass (e.g., core)
# Here we place a red default _before_ the blue default and export the
# color SASS variable as a CSS variable _after_ the core
red_layer <- sass_layer(red, rules = ":root{ --color: #{$color}; }")
sass(sass_bundle(core, red_layer))
sass(sass_bundle(core, red_layer, sass_layer(green)))

# Example of merging layers and removing a layer
# Remember to name the layers that are removable
core_layers <- sass_bundle(core, red = red_layer, green = sass_layer(green))
core_layers # pretty printed for console
core_slim <- sass_bundle_remove(core_layers, "red")
sass(core_slim)


# File attachment example: Create a checkboard pattern .png, then
# use it from a sass layer

tmp_png <- tempfile(fileext = ".png")
grDevices::png(filename = tmp_png, width = 20, height = 20,
  bg = "transparent", antialias = "none")
par(mar = rep_len(0,4), xaxs = "i", yaxs = "i")
plot.new()
rect(c(0,0.5), c(0,0.5), c(0.5,1), c(0.5,1), col = "#00000044", border=NA)
dev.off()

layer <- sass_layer(
  rules = ".bg-check { background-image: url(images/demo_checkboard_bg.png) }",
  file_attachments = c("images/demo_checkboard_bg.png" = tmp_png)
)

output_path <- tempfile(fileext = ".css")
sass(layer, output = output_path, write_attachments = TRUE)
}
