/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.integration.ndk

import com.android.build.gradle.integration.common.category.DeviceTests
import com.android.build.gradle.integration.common.fixture.GradleTestProject
import com.android.build.gradle.integration.common.fixture.app.HelloWorldJniApp
import com.android.builder.core.BuilderConstants
import groovy.transform.CompileStatic
import org.junit.AfterClass
import org.junit.BeforeClass
import org.junit.ClassRule
import org.junit.Test
import org.junit.experimental.categories.Category

import java.util.zip.ZipFile

import static org.junit.Assert.assertNotNull
import static org.junit.Assert.assertNull

/**
 * Integration test of the native plugin with multiple variants without using splits.
 */
@CompileStatic
class NoSplitNdkVariantsTest {

    @ClassRule
    static public GradleTestProject project = GradleTestProject.builder()
            .fromTestApp(new HelloWorldJniApp())
            .addGradleProperties("android.useDeprecatedNdk=true")
            .create()

    @BeforeClass
    static public void setUp() {
        project.getBuildFile() << """
apply plugin: 'com.android.application'

android {
    compileSdkVersion $GradleTestProject.DEFAULT_COMPILE_SDK_VERSION
    buildToolsVersion "$GradleTestProject.DEFAULT_BUILD_TOOL_VERSION"
    defaultConfig {
        ndk {
            moduleName "hello-jni"
        }
    }
    buildTypes {
        release
        debug {
            jniDebuggable true
        }
    }
    productFlavors {
        x86 {
            ndk {
                abiFilter "x86"
            }
        }
        arm {
            ndk {
                abiFilters "armeabi-v7a", "armeabi"
            }
        }
        mips {
            ndk {
                abiFilter "mips"
            }
        }
    }
}
"""
    }

    @AfterClass
    static void cleanUp() {
        project = null
    }

    @Test
    public void assembleX86Release() {
        project.execute("assembleX86Release")

        // Verify .so are built for all platform.
        ZipFile apk = new ZipFile(project.getApk("x86", "release", "unsigned"))
        assertNotNull(apk.getEntry("lib/x86/libhello-jni.so"))
        assertNull(apk.getEntry("lib/mips/libhello-jni.so"))
        assertNull(apk.getEntry("lib/armeabi/libhello-jni.so"))
        assertNull(apk.getEntry("lib/armeabi-v7a/libhello-jni.so"))
    }

    @Test
    public void assembleArmRelease() {
        project.execute("assembleArmRelease")

        // Verify .so are built for all platform.
        ZipFile apk = new ZipFile(project.getApk("arm", "release", "unsigned"))
        assertNull(apk.getEntry("lib/x86/libhello-jni.so"))
        assertNull(apk.getEntry("lib/mips/libhello-jni.so"))
        assertNotNull(apk.getEntry("lib/armeabi/libhello-jni.so"))
        assertNotNull(apk.getEntry("lib/armeabi-v7a/libhello-jni.so"))
    }

    @Test
    public void assembleMipsRelease() {
        project.execute("assembleMipsRelease")

        // Verify .so are built for all platform.
        ZipFile apk = new ZipFile(project.getApk("mips", "release", "unsigned"))
        assertNull(apk.getEntry("lib/x86/libhello-jni.so"))
        assertNotNull(apk.getEntry("lib/mips/libhello-jni.so"))
        assertNull(apk.getEntry("lib/armeabi/libhello-jni.so"))
        assertNull(apk.getEntry("lib/armeabi-v7a/libhello-jni.so"))
    }

    @Test
    @Category(DeviceTests.class)
    public void connectedAndroidTest() {
        if (GradleTestProject.DEVICE_PROVIDER_NAME.equals(BuilderConstants.CONNECTED)) {
            project.execute(GradleTestProject.DEVICE_PROVIDER_NAME + "ArmDebugAndroidTest")
        } else {
            project.execute(GradleTestProject.DEVICE_PROVIDER_NAME + "X86DebugAndroidTest")
        }
    }
}
