% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mahalanobis_D.R
\name{mahalanobis_d}
\alias{mahalanobis_d}
\title{Mahalanobis' \emph{D} (a multivariate Cohen's \emph{d})}
\usage{
mahalanobis_d(
  x,
  y = NULL,
  data = NULL,
  pooled_cov = TRUE,
  mu = 0,
  ci = 0.95,
  alternative = "greater",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x, y}{A data frame or matrix. Any incomplete observations (with \code{NA}
values) are dropped. \code{x} can also be a formula (see details) in which case
\code{y} is ignored.}

\item{data}{An optional data frame containing the variables.}

\item{pooled_cov}{Should equal covariance be assumed? Currently only
\code{pooled_cov = TRUE} is supported.}

\item{mu}{A named list/vector of the true difference in means for each
variable. Can also be a vector of length 1, which will be recycled.}

\item{ci}{Confidence Interval (CI) level}

\item{alternative}{a character string specifying the alternative hypothesis;
Controls the type of CI returned: \code{"two.sided"} (default, two-sided CI),
\code{"greater"} or \code{"less"} (one-sided CI). Partial matching is allowed (e.g.,
\code{"g"}, \code{"l"}, \code{"two"}...). See \emph{One-Sided CIs} in \link{effectsize_CIs}.}

\item{verbose}{Toggle warnings and messages on or off.}

\item{...}{Not used.}
}
\value{
A data frame with the \code{Mahalanobis_D} and potentially its CI
(\code{CI_low} and \code{CI_high}).
}
\description{
Compute effect size indices for standardized difference between two normal
multivariate distributions or between one multivariate distribution and a
defined point. This is the standardized effect size for Hotelling's \eqn{T^2}
test (e.g., \code{DescTools::HotellingsT2Test()}). \emph{D} is computed as:
\cr\cr
\deqn{D = \sqrt{(\bar{X}_1-\bar{X}_2-\mu)^T \Sigma_p^{-1} (\bar{X}_1-\bar{X}_2-\mu)}}
\cr\cr
Where \eqn{\bar{X}_i} are the column means, \eqn{\Sigma_p} is the \emph{pooled}
covariance matrix, and \eqn{\mu} is a vector of the null differences for each
variable. When there is only one variate, this formula reduces to Cohen's
\emph{d}.
}
\details{
To specify a \code{x} as a formula:
\itemize{
\item Two sample case: \code{DV1 + DV2 ~ group} or \code{cbind(DV1, DV2) ~ group}
\item One sample case: \code{DV1 + DV2 ~ 1} or \code{cbind(DV1, DV2) ~ 1}
}
}
\section{Confidence (Compatibility) Intervals (CIs)}{
Unless stated otherwise, confidence (compatibility) intervals (CIs) are
estimated using the noncentrality parameter method (also called the "pivot
method"). This method finds the noncentrality parameter ("\emph{ncp}") of a
noncentral \emph{t}, \emph{F}, or \eqn{\chi^2} distribution that places the observed
\emph{t}, \emph{F}, or \eqn{\chi^2} test statistic at the desired probability point of
the distribution. For example, if the observed \emph{t} statistic is 2.0, with 50
degrees of freedom, for which cumulative noncentral \emph{t} distribution is \emph{t} =
2.0 the .025 quantile (answer: the noncentral \emph{t} distribution with \emph{ncp} =
.04)? After estimating these confidence bounds on the \emph{ncp}, they are
converted into the effect size metric to obtain a confidence interval for the
effect size (Steiger, 2004).
\cr\cr
For additional details on estimation and troubleshooting, see \link{effectsize_CIs}.
}

\section{CIs and Significance Tests}{
"Confidence intervals on measures of effect size convey all the information
in a hypothesis test, and more." (Steiger, 2004). Confidence (compatibility)
intervals and p values are complementary summaries of parameter uncertainty
given the observed data. A dichotomous hypothesis test could be performed
with either a CI or a p value. The 100 (1 - \eqn{\alpha})\% confidence
interval contains all of the parameter values for which \emph{p} > \eqn{\alpha}
for the current data and model. For example, a 95\% confidence interval
contains all of the values for which p > .05.
\cr\cr
Note that a confidence interval including 0 \emph{does not} indicate that the null
(no effect) is true. Rather, it suggests that the observed data together with
the model and its assumptions combined do not provided clear evidence against
a parameter value of 0 (same as with any other value in the interval), with
the level of this evidence defined by the chosen \eqn{\alpha} level (Rafi &
Greenland, 2020; Schweder & Hjort, 2016; Xie & Singh, 2013). To infer no
effect, additional judgments about what parameter values are "close enough"
to 0 to be negligible are needed ("equivalence testing"; Bauer & Kiesser,
1996).
}

\section{Plotting with \code{see}}{

The \code{see} package contains relevant plotting functions. See the \href{https://easystats.github.io/see/articles/effectsize.html}{plotting vignette in the \code{see} package}.
}

\examples{
## Two samples --------------
mtcars_am0 <- subset(mtcars, am == 0,
  select = c(mpg, hp, cyl)
)
mtcars_am1 <- subset(mtcars, am == 1,
  select = c(mpg, hp, cyl)
)

mahalanobis_d(mtcars_am0, mtcars_am1)

# Or
mahalanobis_d(mpg + hp + cyl ~ am, data = mtcars)

mahalanobis_d(mpg + hp + cyl ~ am, data = mtcars, alternative = "two.sided")

# Different mu:
mahalanobis_d(mpg + hp + cyl ~ am,
  data = mtcars,
  mu = c(mpg = -4, hp = 15, cyl = 0)
)


# D is a multivariate d, so when only 1 variate is provided:
mahalanobis_d(hp ~ am, data = mtcars)

cohens_d(hp ~ am, data = mtcars)


# One sample ---------------------------
mahalanobis_d(mtcars[, c("mpg", "hp", "cyl")])

# Or
mahalanobis_d(mpg + hp + cyl ~ 1,
  data = mtcars,
  mu = c(mpg = 15, hp = 5, cyl = 3)
)

}
\references{
\itemize{
\item Del Giudice, M. (2017). Heterogeneity coefficients for Mahalanobis' D as a multivariate effect size. Multivariate Behavioral Research, 52(2), 216-221.
\item Mahalanobis, P. C. (1936). On the generalized distance in statistics. National Institute of Science of India.
\item Reiser, B. (2001). Confidence intervals for the Mahalanobis distance. Communications in Statistics-Simulation and Computation, 30(1), 37-45.
}
}
\seealso{
\code{\link[stats:mahalanobis]{stats::mahalanobis()}}, \code{\link[=cov_pooled]{cov_pooled()}}

Other standardized differences: 
\code{\link{cohens_d}()},
\code{\link{means_ratio}()},
\code{\link{p_superiority}()},
\code{\link{rank_biserial}()}
}
\concept{standardized differences}
